#Django core bits
from django.shortcuts import render_to_response, get_object_or_404, render
from django.template import RequestContext, loader, Context
from django.contrib.auth.decorators import login_required, permission_required
from django.db.models import Q
from django.conf import settings
from django.http import HttpResponseRedirect, Http404, HttpResponse
from django.core.urlresolvers import reverse
from django.contrib.auth.models import User, Group
from django.contrib import messages
from django.template.defaultfilters import slugify
from django.forms.models import modelformset_factory
from datetime import datetime, date, time
import csv
from docx import Document

from modules.core.decorators import superuser_only
from modules.core.functions import *

#Forms
from forms import *

#Models
from models import *
from modules.members.models import *

import emails

#from modules.notifications import signals

@permission_required('admin_users.can_access_abstracts')
def abstract_requests(request):

    enabled_abstract_requests = AbstractRequest.objects.filter(enabled=True).order_by('start_date')
    disabled_abstract_requests = AbstractRequest.objects.filter(enabled=False).order_by('start_date')

    return render_to_response('admin/abstracts/abstract-requests.html',{'enabled_abstract_requests':enabled_abstract_requests,'disabled_abstract_requests':disabled_abstract_requests},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_abstracts')
def add_abstract_request(request):

    abstract_request_form = AbstractRequestForm()

    if request.POST:
        abstract_request_form = AbstractRequestForm(request.POST)

        if abstract_request_form.is_valid():
            try:
                abstract_request = abstract_request_form.save(commit=False)
                abstract_request.slug = slugify_unique(abstract_request.name,AbstractRequest,'slug')
                abstract_request.save()

                messages.success(request,'Abstract Request Added')

                return HttpResponseRedirect(reverse('admin_abstracts'))

            except:
                messages.error(request,'Could not add Abstract Request')

    return render_to_response('admin/abstracts/add-abstract-request.html',{'abstract_request_form':abstract_request_form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_abstracts')
def edit_abstract_request(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    abstract_request_form = AbstractRequestForm(instance=abstract_request)

    if request.POST:
        abstract_request_form = AbstractRequestForm(request.POST,instance=abstract_request)

        if abstract_request_form.is_valid():
            try:
                abstract_request_form.save()

                messages.success(request,'Abstract Request Updated')

                return HttpResponseRedirect(reverse('admin_abstracts'))

            except:
                messages.error(request,'Could not update Abstract Request')

    return render_to_response('admin/abstracts/edit-abstract-request.html',{'abstract_request_form':abstract_request_form,'abstract_request':abstract_request},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_abstracts')
def delete_abstract_request(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)

    try:
        abstract_request.delete()
        messages.success(request,'Abstract Request Deleted')
    except:
        messages.error(request,'Could not delete Abstract Request')

    return HttpResponseRedirect(reverse('admin_abstracts'))

@permission_required('admin_users.can_access_abstracts')
def enable_abstract_request(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)

    try:
        abstract_request.enabled = True
        abstract_request.save()
        messages.success(request,'Abstract Request Enabled')
    except:
        messages.error(request,'Could not enable Abstract Request')

    return HttpResponseRedirect(reverse('admin_abstracts'))

@permission_required('admin_users.can_access_abstracts')
def disable_abstract_request(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)

    try:
        abstract_request.enabled = False
        abstract_request.save()
        messages.success(request,'Abstract Request Disabled')
    except:
        messages.error(request,'Could not disable Abstract Request')

    return HttpResponseRedirect(reverse('admin_abstracts'))

@permission_required('admin_users.can_access_abstracts')
def submissions(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True).order_by('-created')

    return render_to_response('admin/abstracts/submissions.html',{'abstract_request':abstract_request,'content_submissions':content_submissions},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_abstracts')
def load_submission(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)

    if request.POST:
        if request.POST.get('unique_code'):
            try:
                content_submission = AbstractSubmission.objects.get(unique_code=request.POST['unique_code'],abstract_request=abstract_request)

                return HttpResponseRedirect(reverse('admin_abstracts_view_submission',args=[abstract_request.id,content_submission.id]))

            except AbstractSubmission.DoesNotExist:
                messages.error(request,'Submission with that Unique Code not found.')


    return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))

@permission_required('admin_users.can_access_abstracts')
def add_all_submissions_review_group(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True).order_by('-created')

    add_submission_review_form = AddSubmissionReviewGroupForm()

    if request.POST:
        add_submission_review_form = AddSubmissionReviewGroupForm(request.POST)

        if add_submission_review_form.is_valid():
            group = get_object_or_404(MemberGroup,id=request.POST['group'])

            if request.POST['type']:
                types = add_submission_review_form.cleaned_data['type']
                content_submissions = content_submissions.filter(content_type__in=types)

            if request.POST['category']:
                categories = add_submission_review_form.cleaned_data['category']
                content_submissions = content_submissions.filter(abstract_category__in=categories)

            for content_submission in content_submissions:
                members = group.members.all()
                for member in members:
                    if member.user.email != content_submission.email:
                        try:
                            current_review = AbstractSubmissionReview.objects.filter(reviewer=member,content_submission=content_submission)
                            if not current_review:
                                review = AbstractSubmissionReview(reviewer=member,content_submission=content_submission)
                                review.save()
                                messages.success(request,'ID %s, Request to review saved for %s' % (content_submission.unique_code,member))
                        except:
                            messages.error(request,'ID %s, Could not save Review for %s' % (content_submission.unique_code,member))
                    else:
                        messages.warning(request,'ID %s, did not add Review Request for %s, is same Person' % (content_submission.unique_code,member))

            return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))

    return render_to_response('admin/abstracts/add-all-submissions-review-group.html',{'abstract_request':abstract_request,'add_submission_review_form':add_submission_review_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_abstracts')
def delete_all_submissions_review_group(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True).order_by('-created')

    delete_submission_review_form = DeleteSubmissionReviewGroupForm()

    if request.POST:
        delete_submission_review_form = DeleteSubmissionReviewGroupForm(request.POST)

        if delete_submission_review_form.is_valid():
            group = get_object_or_404(MemberGroup,id=request.POST['group'])

            for content_submission in content_submissions:
                members = group.members.all()
                for member in members:
                    current_review = AbstractSubmissionReview.objects.filter(reviewer=member,content_submission=content_submission,sent=False,reviewed=False)
                    if current_review:
                        for review in current_review:
                            review.delete()

            messages.success(request,'Requests to review deleted')

            return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))

    return render_to_response('admin/abstracts/delete-all-submissions-review-group.html',{'abstract_request':abstract_request,'delete_submission_review_form':delete_submission_review_form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_abstracts')
def send_all_review_requests(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True).order_by('-created')

    sent_reviewers = []

    for content_submission in content_submissions:

        pending_reviews = AbstractSubmissionReview.objects.filter(content_submission=content_submission,sent=False).order_by('-created')

        for review in pending_reviews:

            #try:

            review.sent = True
            review.save()

            if review.reviewer not in sent_reviewers:

                member = get_object_or_404(Member,id=review.reviewer.id)

                activity = UserActivity(member=member,type='abstract-notification',title='Abstract Review Required',text='You have been selected to review submissions for %s, to review this content please go <a href="%s">here</a>.' % (content_submission.abstract_request,reverse('review_content_requests_abstract_request',args=[abstract_request.slug])))
                activity.save()

                #signal here
                emails.abstract_content_review_requested_multiple(request=request,content_submission_review=review)

                sent_reviewers.append(review.reviewer)

            messages.success(request,'ID %s, Review Request sent to %s %s' % (content_submission.unique_code,review.reviewer.given_name,review.reviewer.surname))

            #except:
            #    review.sent = False
            #    review.save()
            #    messages.error(request,'ID %s, Could not send Request to %s %s' % (content_submission.unique_code,review.reviewer.given_name,review.reviewer.surname))

    return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))


@permission_required('admin_users.can_access_abstracts')
def export_submissions(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True).order_by('-created')

    fields = ['Unique Code', 'Category Submitted', 'Type', 'Score Average', 'Institution', 'Title']

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=content_submissions_anonymous.csv"

    writer = csv.writer(response)
    writer.writerow(fields)

    for submission in content_submissions:

        if submission.get_authors():
            primary_author = submission.get_authors()[0]
            primary_contact_email = primary_author.email
            institution = primary_author.institution
        else:
            primary_contact_email = submission.email
            institution = 'Unknown'

        submission_array = [submission.unique_code, submission.abstract_category, submission.content_type, submission.get_score_average(), institution, submission.abstract_title]

        writer.writerow([unicode(s).encode("utf-8") for s in submission_array])

    return response


@permission_required('admin_users.can_access_abstracts')
def export_submissions_general(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True).order_by('-created')

    fields = ['Unique Code', 'Email Address', 'Institution', 'Department', 'Title', 'Name','Date of Birth','Type']

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=content_submissions_general.csv"

    writer = csv.writer(response)
    writer.writerow(fields)

    for submission in content_submissions:
        if submission.get_authors():
            primary_author = submission.get_authors()[0]
            primary_contact_email = primary_author.email
            institution = primary_author.institution
        else:
            primary_contact_email = submission.email
            institution = 'Unknown'

        title = submission.abstract_title

        if submission.get_authors():
            primary_author = submission.get_authors()[0]
            primary_contact_name = primary_author.name
            primary_contact_email = primary_author.email
            institution = primary_author.institution
            department = primary_author.department
        else:
            primary_contact_name = "%s %s" % (submission.given_name,submission.surname)
            primary_contact_email = submission.email
            institution = 'Unknown'
            department = 'Unknown'

        submission_array = [submission.unique_code, primary_contact_email, institution, department, title, primary_contact_name,submission.date_of_birth,submission.content_type]

        writer.writerow([unicode(s).encode("utf-8") for s in submission_array])

    return response


@permission_required('admin_users.can_access_abstracts')
def export_submissions_reviews(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_reviews = AbstractSubmissionReview.objects.filter(content_submission__abstract_request=abstract_request,reviewed=True).order_by('-reviewed','-content_submission__created')

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=submission_reviews.csv"
    writer = csv.writer(response)

    fields = ['Title','Given Name','Surname','Email Address','Telephone','Submitted','Content Type','Abstract Title','Abstract Category','Authors','Abstract Content','Review By','Review Created','Review Score','Review Comment']

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=content_submissions_reviews.csv"

    writer = csv.writer(response)
    writer.writerow(fields)

    for review in content_reviews:

        content_submission = review.content_submission

        authors = ''
        author_count = 1
        for author in content_submission.get_authors():
            authors = '%s Author %s: Name: %s, Status: %s, Institution: %s, Email: %s, Telephone: %s,' % (authors,author_count,author.name,author.get_status_display(),author.institution,author.email,author.telephone)
            author_count = author_count + 1

        if review.suitable_for_poster:
            suitable_for_poster = "Yes"
        else :
            suitable_for_poster = "No"


        submission_array = [content_submission.title,content_submission.given_name,content_submission.surname,content_submission.email,content_submission.telephone,content_submission.created,content_submission,review.content_submission.content_type,content_submission.abstract_title,content_submission.abstract_category,authors,content_submission.abstract_content,review.reviewer,review.review_created,review.get_total(),review.comment]

        writer.writerow([unicode(s).encode("utf-8") for s in submission_array])

    return response


@permission_required('admin_users.can_access_abstracts')
def approved_submissions(request,abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,status='approved')

    return render(request,'admin/abstracts/approved-submissions.html',{'abstract_request':abstract_request,'content_submissions':content_submissions})


@permission_required('admin_users.can_access_abstracts')
def view_submission(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)
    submission_files = content_submission.get_files()
    submission_authors = content_submission.get_authors()
    submission_reviews = AbstractSubmissionReview.objects.filter(content_submission=content_submission).order_by('-created')

    return render(request,'admin/abstracts/view-content-submission.html',{'abstract_request':abstract_request,'content_submission':content_submission,'submission_files':submission_files,'submission_authors':submission_authors,'submission_reviews':submission_reviews})

@permission_required('admin_users.can_access_abstracts')
def edit_submission(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    edit_content_submission_form = EditAbstractSubmissionForm(instance=content_submission)

    if request.POST:

        edit_content_submission_form = EditAbstractSubmissionForm(request.POST,instance=content_submission)

        if edit_content_submission_form.is_valid():

            try:
                edit_content_submission_form.save()

                messages.success(request,'Submission has been saved')
                return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))

            except:
                messages.error(request,'Could not save submission')

    return render_to_response('admin/abstracts/edit-content-submission.html',{'abstract_request':abstract_request,'content_submission':content_submission,'edit_content_submission_form':edit_content_submission_form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_abstracts')
def delete_submission(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    try:
        content_submission.delete()
        messages.success(request,'Submission has been deleted')
    except:
        messages.error(request,'Could not delete submission')


    return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))

@permission_required('admin_users.can_access_abstracts')
def approve_submission(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    try:
        content_submission.status = 'approved'
        content_submission.save()
        messages.success(request,'Submission has been approved')
    except:
        messages.error(request,'Could not approve submission')


    return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))

@permission_required('admin_users.can_access_abstracts')
def reject_submission(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    try:
        content_submission.status = 'rejected'
        content_submission.save()
        messages.success(request,'Submission has been approved')
    except:
        messages.error(request,'Could not approve submission')


    return HttpResponseRedirect(reverse('admin_abstracts_submissions',args=[abstract_request.id]))


@permission_required('admin_users.can_access_abstracts')
def submission_reviews(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)
    submission_reviews = AbstractSubmissionReview.objects.filter(content_submission=content_submission).order_by('-created')

    unsent_reviews = AbstractSubmissionReview.objects.filter(content_submission=content_submission,sent=False)
    completed_reviews = AbstractSubmissionReview.objects.filter(content_submission=content_submission,reviewed=True)


    return render_to_response('admin/abstracts/submission-reviews.html',{'abstract_request':abstract_request,'content_submission':content_submission,'submission_reviews':submission_reviews,'unsent_reviews':unsent_reviews,'completed_reviews':completed_reviews},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_abstracts')
def send_review_requests(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)
    pending_reviews = AbstractSubmissionReview.objects.filter(content_submission=content_submission,sent=False).order_by('-created')

    for review in pending_reviews:

        #try:

        review.sent = True
        review.save()

        member = get_object_or_404(Member,id=review.reviewer.id)

        activity = UserActivity(member=member,type='abstract-notification',title='Abstract Submission Review Required',text='You have been selected as a reviewer for submission %s for %s, to review this submission please go <a href="%s">here</a>.' % (content_submission.unique_code,content_submission.abstract_request,reverse('review_content_request',args=[abstract_request.slug,review.id])))
        activity.save()

        #signal here
        emails.abstract_content_review_requested_multiple(request=request,content_submission_review=review)

        messages.success(request,'Review Request sent to %s %s' % (review.reviewer.given_name,review.reviewer.surname))

        #except:
        #    review.sent = False
        #    review.save()

        #    messages.error(request,'Could not send Request to %s %s' % (review.reviewer.given_name,review.reviewer.surname))

    return HttpResponseRedirect(reverse('admin_abstracts_submission_reviews',args=[abstract_request.id,content_submission.id]))


@permission_required('admin_users.can_access_abstracts')
def add_submission_review(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    add_submission_review_form = AddSubmissionReviewForm()

    if request.POST:
        add_submission_review_form = AddSubmissionReviewForm(request.POST)

        if add_submission_review_form.is_valid():
            try:
                review = add_submission_review_form.save(commit=False)
                review.content_submission = content_submission
                review.save()

                messages.success(request,'Request to review saved')

                return HttpResponseRedirect(reverse('admin_abstracts_submission_reviews',args=[abstract_request.id,content_submission.id]))

            except:
                messages.error(request,'Could not save Review')

    return render_to_response('admin/abstracts/add-submission-review.html',{'abstract_request':abstract_request,'content_submission':content_submission,'add_submission_review_form':add_submission_review_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_abstracts')
def add_submission_review_group(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    add_submission_review_form = AddSubmissionReviewGroupForm()

    if request.POST:
        add_submission_review_form = AddSubmissionReviewGroupForm(request.POST)

        if add_submission_review_form.is_valid():
            group = get_object_or_404(MemberGroup,id=request.POST['group'])

            members = group.members.all()
            for member in members:
                try:
                    current_review = AbstractSubmissionReview.objects.filter(reviewer=member,content_submission=content_submission)
                    if not current_review:

                        review = AbstractSubmissionReview(reviewer=member,content_submission=content_submission)
                        review.save()

                        messages.success(request,'Request to review saved for %s' % (member))

                except:
                    messages.error(request,'Could not save Review for %s' % (member))

            return HttpResponseRedirect(reverse('admin_abstracts_submission_reviews',args=[abstract_request.id,content_submission.id]))

    return render_to_response('admin/abstracts/add-submission-review-group.html',{'abstract_request':abstract_request,'content_submission':content_submission,'add_submission_review_form':add_submission_review_form},context_instance=RequestContext(request))



@permission_required('admin_users.can_access_abstracts')
def view_submission_review(request,abstract_id,submission_id,review_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)
    submission_review = get_object_or_404(AbstractSubmissionReview,id=review_id,content_submission=content_submission)

    return render_to_response('admin/abstracts/view-submission-review.html',{'abstract_request':abstract_request,'content_submission':content_submission,'submission_review':submission_review},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_abstracts')
def delete_submission_review(request,abstract_id,submission_id,review_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    submission_review = get_object_or_404(AbstractSubmissionReview,id=review_id,content_submission=content_submission)

    try:
        submission_review.delete()
        messages.success(request,'Request to review deleted')
    except:
        messages.error(request,'Could not delete request to review')

    return HttpResponseRedirect(reverse('admin_abstracts_submission_reviews',args=[abstract_request.id,content_submission.id]))


@permission_required('admin_users.can_access_abstracts')
def download_reviews(request,abstract_id,submission_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    content_submission = get_object_or_404(AbstractSubmission,id=submission_id,abstract_request=abstract_request)

    reviews = AbstractSubmissionReview.objects.filter(content_submission=content_submission,reviewed=True).order_by('-review_created')

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=submission_reviews.csv"
    writer = csv.writer(response)

    fields = ['Submission Code','Category','Scorer','Time Submitted','Title Score', 'Abstract Construction','Content Score','Relevance Score','Priority Score','Total','Comment']
    writer.writerow(fields)

    for review in reviews:

        review_array = [content_submission.unique_code,content_submission.abstract_category,review.reviewer,review.review_created,review.title_score,review.abstract_construction,review.content_score,review.relevance_score,review.priority_score,review.get_total(),review.comment]

        writer.writerow([unicode(s).encode("utf-8") for s in review_array])

    return response


@permission_required('admin_users.can_access_abstracts')
def submissions_approve(request, abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    submissions = (AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True).order_by('-created'))

    if request.method == 'POST':

        for submission in submissions:

            if request.POST.get('submission_%s_status' % (submission.id)):
                if submission.status == 'pending':
                    submission.status = request.POST['submission_%s_status' % (submission.id)]
                    submission.save()

                    if submission.status == 'approved':
                        emails.abstract_content_submission_approved(request=request,content_submission=submission)
                    elif submission.status == 'rejected':
                        emails.abstract_content_submission_rejected(request=request,content_submission=submission)

        messages.success(request, 'Submission statuses have been changed')

        return HttpResponseRedirect(reverse('admin_abstracts_approve',args=[abstract_request.id]))

    return render_to_response('admin/abstracts/approve.html',{'abstract_request':abstract_request,'submissions':submissions},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_abstracts')
def export_docx(request, abstract_id):

    abstract_request = get_object_or_404(AbstractRequest,id=abstract_id)
    abstract_submissions = AbstractSubmission.objects.filter(abstract_request=abstract_request,complete=True)
    #,status='approved')
    content_types = AbstractContentType.objects.all()

    doc = Document()

    for content_type in content_types:

        if content_type.get_submissions(abstract_request):

            doc.add_heading('%s\n' % content_type.name, level=1)

            for submission in content_type.get_submissions(abstract_request):

                doc.add_heading(submission.abstract_title,level=2)
                p = doc.add_paragraph()
                authors = submission.get_authors()
                if authors:
                    authors_string = "Authors: "
                    for author in authors:
                        authors_string = "%s%s - %s %s," % (authors_string,author.name,author.department,author.institution)
                    authors_string = "%s\n" % (authors_string)
                    p.add_run(authors_string)
                else:
                    p.add_run("Author: %s\n" % (submission.get_name()))

                p.add_run("Category: %s" % (submission.abstract_category))

                doc.add_paragraph(submission.abstract_content)
                doc.add_paragraph()

    doc_name = '%s Submissions.docx' % (abstract_request.name)
    doc_location = '%s/submissions/%s' %(settings.PRIVATE_FILES,doc_name)
    doc.save(doc_location)

    with open(doc_location) as f:
        data = f.read()

    response = HttpResponse(data, content_type='application/vnd.openxmlformats-officedocument.wordprocessingml.document')
    response['Content-Disposition'] = 'attachment; filename=%s' % (doc_name)
    return response


def content_types(request):

    content_types = AbstractContentType.objects.all()

    return render(request,'admin/abstracts/content-types/content-types.html',{'content_types':content_types})


def add_content_type(request):

    content_type_form = AbstractContentTypeForm()

    if request.POST:
        content_type_form = AbstractContentTypeForm(request.POST)

        if content_type_form.is_valid():
            try:
                content_type_form.save()

                messages.success(request,'Content type Created')

                return HttpResponseRedirect(reverse('admin_abstracts_content_types'))

            except:
                messages.errror(request,'Could not create Content Type')

    return render(request,'admin/abstracts/content-types/add-content-type.html',{'content_type_form':content_type_form})

def edit_content_type(request,type_id):

    content_type = get_object_or_404(AbstractContentType,id=type_id)
    content_type_form  = AbstractContentTypeForm(instance=content_type)

    if request.POST:
        content_type_form = AbstractContentTypeForm(request.POST,instance=content_type)

        if content_type_form.is_valid():
            try:
                content_type_form.save()
                messages.success(request,'Content Type Updated')
                return HttpResponseRedirect(reverse('admin_abstracts_content_types'))

            except:
                messages.error(request,'Could not update Content Type')

    return render(request,'admin/abstracts/content-types/edit-content-type.html',{'content_type':content_type,'content_type_form':content_type_form})

def delete_content_type(request,type_id):

    content_type = get_object_or_404(AbstractContentType,id=type_id)

    try:
        content_type.delete()
        messages.success(request,'Content Type Deleted')
    except:
        messages.error(request,'Could not Delete Content Type')

    return HttpResponseRedirect(reverse('admin_abstracts_content_types'))

def categories(request):

    categories = AbstractCategory.objects.all()

    return render(request,'admin/abstracts/categories/categories.html',{'categories':categories})


def add_category(request):

    category_form = AbstractCategoryForm()

    if request.POST:
        category_form = AbstractCategoryForm(request.POST)

        if category_form.is_valid():
            try:
                category_form.save()

                messages.success(request,'Category Created')

                return HttpResponseRedirect(reverse('admin_abstracts_categories'))

            except:
                messages.error(request,'Could not create Category')

    return render(request,'admin/abstracts/categories/add-category.html',{'category_form':category_form})

def edit_category(request,category_id):

    category = get_object_or_404(AbstractCategory,id=category_id)
    category_form  = AbstractCategoryForm(instance=category)

    if request.POST:
        content_type_form = AbstractCategoryForm(request.POST,instance=category)

        if content_type_form.is_valid():
            try:
                content_type_form.save()
                messages.success(request,'Category Updated')
                return HttpResponseRedirect(reverse('admin_abstracts_categories'))

            except:
                messages.error(request,'Could not update Category')

    return render(request,'admin/abstracts/categories/edit-category.html',{'category':category,'category_form':category_form})

def delete_category(request,category_id):

    category = get_object_or_404(AbstractCategory,id=category_id)

    try:
        category.delete()
        messages.success(request,'Category Deleted')
    except:
        messages.error(request,'Could not Delete Category')

    return HttpResponseRedirect(reverse('admin_abstracts_categories'))
