#Django core bits
from django.shortcuts import render_to_response, get_object_or_404
from django.template import RequestContext, loader, Context
from django.contrib import messages
from django.contrib.auth import logout
from django.contrib.auth.models import User
from django.contrib.auth.decorators import login_required, permission_required
from django.views.decorators.csrf import csrf_exempt
from django.db.models import Q
from django.conf import settings
from django.http import HttpResponseRedirect, Http404, HttpResponse
from django.core.urlresolvers import reverse

from modules.core.decorators import *
from modules.core.functions import *

from models import *
from forms import *

@permission_required('admin_users.can_access_blocks')
def homepage_content(request):
    
    try:
        homepage_settings = HomepageSettings.objects.get(id=1)
    except:
        homepage_settings = False
        
    if homepage_settings:    
        homepage_content_form = HomepageContentForm(instance=homepage_settings)
    else:
        homepage_content_form = HomepageContentForm()
        
    if request.POST:
        
        if homepage_settings:    
            homepage_content_form = HomepageContentForm(request.POST,request.FILES,instance=homepage_settings)
        else:
            homepage_content_form = HomepageContentForm(request.POST,request.FILES)
        
        if homepage_content_form.is_valid():
            
            try:
                homepage_content_form.save()
                
                messages.success(request,'Homepage Content Updated')
                
                return HttpResponseRedirect(reverse('admin_blocks_homepage_content'))
                
            except:
                messages.error(request,'Could not update Homepage Content')
            
    return render_to_response('admin/homepage_content/edit-homepage-content.html',{'homepage_content_form':homepage_content_form,'homepage_settings':homepage_settings},context_instance=RequestContext(request))
    
    
@permission_required('admin_users.can_access_blocks')
def homepage_blocks(request):
    
    homepage_blocks = HomepageBlock.objects.all().order_by('block_type','name')
    
    return render_to_response('admin/homepage_blocks/blocks.html',{'homepage_blocks':homepage_blocks},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_blocks')
def homepage_blocks_layout(request):
    
    try:
        homepage_settings = HomepageSettings.objects.get(id=1)
    except:
        raise Http404
    
    block_layouts = BlockLayout.objects.all()
    
    return render_to_response('admin/homepage_blocks/change-layout.html',{'homepage_settings':homepage_settings,'block_layouts':block_layouts},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_blocks')
def homepage_blocks_layout_load(request,layout_id):
    
    block_layout = get_object_or_404(BlockLayout,id=layout_id)
    
    try:
        homepage_settings = HomepageSettings.objects.get(id=1)
    except:
        raise Http404
        
    if block_layout == homepage_settings.block_layout:
        blocks = HomepageBlock.objects.filter(active=False)
    else:
        blocks = HomepageBlock.objects.all()

    return render_to_response('admin/homepage_blocks/partials/load-layout.html',{'block_layout':block_layout,'blocks':blocks},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_blocks')
@csrf_exempt
def homepage_blocks_layout_save(request,layout_id):
    
    block_layout = get_object_or_404(BlockLayout,id=layout_id)
    
    try:
        homepage_settings = HomepageSettings.objects.get(id=1)
    except:
        raise Http404
    
    elements = HomepageBlock.objects.filter(active=True)
    for element in elements:
        element.active = False
        element.save()
    
    if request.POST:
        
        for key,value in request.POST.items():
            element = get_object_or_404(HomepageBlock,id=value)
            element.active = True
            element.location = key
            element.block_layout = block_layout
            element.save()
        
        homepage_settings.block_layout = block_layout
        homepage_settings.save()
        
        return HttpResponse('Ok')

    else :
        raise Exception('not post')

    raise Http404

@permission_required('admin_users.can_access_blocks')
def add_block(request):
    
    homepage_block_form = HomepageBlockAddForm()
    
    if request.POST:
        
        homepage_block_form = HomepageBlockAddForm(request.POST,request.FILES)
        
        try:
            homepage_block_form.save()
            messages.success(request,'Block has been created')
        
            return HttpResponseRedirect(reverse('admin_blocks_homepage_blocks'))
    
        except:
            messages.error(request,'Could not save block')
            
    return render_to_response('admin/homepage_blocks/add-block.html',{'homepage_block_form':homepage_block_form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_blocks')
def edit_block(request,block_id):
    
    homepage_block = get_object_or_404(HomepageBlock,id=block_id)
    
    homepage_block_form = HomepageBlockEditForm(instance=homepage_block)
    
    if request.POST:
        homepage_block_form = HomepageBlockEditForm(request.POST,request.FILES,instance=homepage_block)
        
        try:
            homepage_block_form.save()
            
            messages.success(request,'Homepage Block Edited')
            
            return HttpResponseRedirect(reverse('admin_blocks_homepage_blocks'))
            
        except:
            messages.error(request,'Could not update Homepage Block')
            
            
    return render_to_response('admin/homepage_blocks/edit-block.html',{'homepage_block':homepage_block,'homepage_block_form':homepage_block_form},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def delete_block(request,block_id):
    
    homepage_block = get_object_or_404(HomepageBlock,id=block_id)
    
    try:
        homepage_block.delete()
        messages.success(request,'Homepage Block has been deleted')
    except:
        messages.error(request,'Could not delete Homepage Block')
        
    return HttpResponseRedirect(reverse('admin_blocks_homepage_blocks'))
    
@permission_required('admin_users.can_access_blocks')
def menus(request):
    
    menus = Menu.objects.all().order_by('name')
    
    return render_to_response('admin/menu/menus.html',{'menus':menus},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def menu_items(request,menu_slug):
    
    menu = get_object_or_404(Menu,slug=menu_slug)
    
    menu_items = MenuItem.objects.filter(parent__isnull=True).order_by('order')
    
    return render_to_response('admin/menu/menu-items.html',{'menu':menu,'menu_items':menu_items},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def add_menu_item(request,menu_slug):
    
    menu = get_object_or_404(Menu,slug=menu_slug)
    
    menu_item_form = MenuItemForm()
    
    if request.POST:
        menu_item_form = MenuItemForm(request.POST)
        
        if menu_item_form.is_valid():
            try:
                menu_item = menu_item_form.save(commit=False)
                menu_item.menu = menu
                menu_item.order = menu.get_next_order()
                menu_item.save()
                
                messages.success(request,'Menu Added')
                
                return HttpResponseRedirect(reverse('admin_blocks_menu_items',args=[menu.slug]))
            
            except:
                messages.error(request,'Could not add menu Item')
                
    return render_to_response('admin/menu/add-item.html',{'menu':menu,'menu_item_form':menu_item_form},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def edit_menu_item(request,menu_slug,item_id):
    
    menu = get_object_or_404(Menu,slug=menu_slug)
    menu_item = get_object_or_404(MenuItem,id=item_id,menu=menu)
    
    menu_item_form = MenuItemForm(instance=menu_item)
    
    if request.POST:
        menu_item_form = MenuItemForm(request.POST,instance=menu_item)
        
        if menu_item_form.is_valid():
            try:
                menu_item_form.save()
                
                messages.success(request,'Menu Saved')
                
                return HttpResponseRedirect(reverse('admin_blocks_menu_items',args=[menu.slug]))
            
            except:
                messages.error(request,'Could not Update menu Item')
                
    return render_to_response('admin/menu/edit-item.html',{'menu':menu,'menu_item':menu_item,'menu_item_form':menu_item_form},context_instance=RequestContext(request))
    
    
@permission_required('admin_users.can_access_blocks')
def delete_menu_item(request,menu_slug,item_id):
    
    menu = get_object_or_404(Menu,slug=menu_slug)
    menu_item = get_object_or_404(MenuItem,id=item_id,menu=menu)
    
    try:
        menu_item.delete()
        
        messages.success(request,'Menu Item Deleted')
        
    except:
        messages.error(request,'Could not delete Menu item')
        
    return HttpResponseRedirect(reverse('admin_blocks_menu_items',args=[menu.slug]))
    
@permission_required('admin_users.can_access_blocks')
@csrf_exempt
def reorder_menu_items(request,menu_slug):
    
    menu = get_object_or_404(Menu,slug=menu_slug)
    
    if request.POST and request.POST.get('order'):
        
        items = request.POST['order'].split('&')
        
        order = 1
        
        for item in items:
            
            item_array = item.split('=')
            item_id = item_array[1]
                        
            item = get_object_or_404(MenuItem,id=item_id)
            item.order = order
            item.save()
            
            order = order + 1
        
        return HttpResponse('Ok')
        
    else:
        raise Http404
        
        
@permission_required('admin_users.can_access_blocks')
def content_blocks(request):
    
    content_blocks = ContentBlock.objects.all().order_by('name')
    
    return render_to_response('admin/content_blocks/content-blocks.html',{'content_blocks':content_blocks},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def edit_content_block(request,block_id):
    
    content_block = get_object_or_404(ContentBlock,id=block_id)
    content_block_form = ContentBlockForm(instance=content_block)
    
    if request.POST:
        content_block_form = ContentBlockForm(request.POST,instance=content_block)
    
        if content_block_form.is_valid():
            try:
                content_block_form.save()
            
                messages.success(request,'Content Block Saved')
            
                return HttpResponseRedirect(reverse('admin_blocks_content_blocks'))
        
            except:
                messages.error(request,'Could not Update Content Block')
                
    return render_to_response('admin/content_blocks/edit-block.html',{'content_block':content_block,'content_block_form':content_block_form},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def homepage_search(request):
    
    audience_search = HomepageSearch.objects.filter(type='audience')
    content_type_search = HomepageSearch.objects.filter(type='content-type')
    
    return render_to_response('admin/homepage_search/homepage-search.html',{'audience_search':audience_search,'content_type_search':content_type_search},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_blocks')
def add_homepage_search(request):
    
    add_homepage_search_form = HomepageSearchForm()
    
    if request.POST:
        add_homepage_search_form = HomepageSearchForm(request.POST)
        
        if add_homepage_search_form.is_valid():
            try:
                homepage_search = add_homepage_search_form.save(commit=False)
                homepage_search.slug = slugify_unique(homepage_search.name,HomepageSearch,'slug')
                homepage_search.save()
            
                messages.success(request,'Homepage Search Created')
        
                return HttpResponseRedirect(reverse('admin_blocks_homepage_search'))
                
            except:
                messages.error(request,'Could not create Homepage Search')
                
    return render_to_response('admin/homepage_search/add-homepage-search.html',{'add_homepage_search_form':add_homepage_search_form},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def edit_homepage_search(request,search_id):
    
    homepage_search = get_object_or_404(HomepageSearch,id=search_id)
    homepage_search_form = HomepageSearchForm(instance=homepage_search)
    
    if request.POST:
        homepage_search_form = HomepageSearchForm(request.POST,instance=homepage_search)
        
        if homepage_search_form.is_valid():
            
            try:
                homepage_search_form.save()
                
                messages.success(request,'Homepage Search Updated')
                
                return HttpResponseRedirect(reverse('admin_blocks_homepage_search'))
                
            except:
                messages.error(request,'Could not update Homepage Search')
                
    return render_to_response('admin/homepage_search/edit-homepage-search.html',{'homepage_search':homepage_search,'homepage_search_form':homepage_search_form},context_instance=RequestContext(request))
    
@permission_required('admin_users.can_access_blocks')
def delete_homepage_search(request,search_id):
    
    homepage_search = get_object_or_404(HomepageSearch,id=search_id)
    
    try:
        homepage_search.delete()
        
        messages.success(request,'Homepage Search Deleted')
        
    except:
        messages.error(request,'Could not delete Homepage Search')
        
    return HttpResponseRedirect(reverse('admin_blocks_homepage_search'))

@permission_required('admin_users.can_access_blocks')
def block_banners(request, block_id):
    block = HomepageBlock.objects.get(pk=block_id)
    banners = BannerImage.objects.filter(block=block)
    return render_to_response('admin/banners/banners.html',{'banners':banners, 'homepage_block': block},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_blocks')
def block_banners_add(request, block_id):
    block = HomepageBlock.objects.get(pk=block_id)
    banner_form = BannerForm()
    
    if request.POST:
        banner_form = BannerForm(request.POST, request.FILES)
        
        if banner_form.is_valid():
            try:
                banner = banner_form.save(commit=False)
                banner.block = block
                banner.save()
                
                messages.success(request,'Banner Added')
                
                return HttpResponseRedirect(reverse('admin_blocks_block_banners',args=[block.id]))
            except:
                messages.error(request,'Could not add banner')
                
    return render_to_response('admin/banners/add-banner.html',{'block_id': block_id, 'banner_form': banner_form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_blocks')
def block_banners_edit(request, block_id, banner_id):
    block = get_object_or_404(HomepageBlock,pk=block_id)
    banner = get_object_or_404(BannerImage,pk=banner_id)
    banner_form = BannerForm(instance=banner)
    
    if request.POST:
        banner_form = BannerForm(request.POST, request.FILES, instance=banner)
        
        if banner_form.is_valid():
            try:
                banner = banner_form.save(commit=False)
                banner.save()
                
                messages.success(request,'Banner Saved')
                
                return HttpResponseRedirect(reverse('admin_blocks_block_banners',args=[block.id]))
            except:
                messages.error(request,'Could not save banner')
                
    return render_to_response('admin/banners/edit-banner.html',{'homepage_block': block, 'banner': banner, 'banner_form': banner_form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_blocks')
def block_banners_delete(request, block_id, banner_id):
    banner = BannerImage.objects.get(id=banner_id)
    
    try:
        banner.delete()
        messages.success(request,'Banner has been deleted')
    except:
        messages.error(request,'Could not delete Banner')
        
    return HttpResponseRedirect(reverse('admin_blocks_block_banners',args=[block_id]))