from django.db import models
from django.db.models import Q

from datetime import datetime,date,time

from django.core.exceptions import ValidationError


class Menu(models.Model):

    name = models.CharField(max_length=200)
    slug = models.SlugField(max_length=200)

    def __unicode__(self):
        return self.name


    def get_top_level(self):
        items = MenuItem.objects.filter(menu=self,parent__isnull=True).order_by('order')
        return items

    def get_next_order(self,parent=False):

        if parent:
            menu_items = MenuItem.object.filter(parent=parent,menu=self).order_by('-order')
        else:
            menu_items = MenuItem.objects.filter(parent__isnull=True,menu=self).order_by('-order')

        if menu_items:
            last_item = menu_items[0]
            order = last_item.order + 1
        else:
            order = 1

        return order


class MenuItem(models.Model):

    menu    = models.ForeignKey('Menu')
    parent  = models.ForeignKey('MenuItem',blank=True,null=True)

    name    = models.CharField(max_length=200)
    url     = models.CharField(max_length=200)

    order   = models.IntegerField(default=0)

    def __unicode__(self):
        return self.name

    def get_children(self):
        children = MenuItem.objects.filter(parent=self).order_by('order')
        return children


class BlockLayout(models.Model):

    name = models.CharField(max_length=200)
    slug = models.SlugField()

    def __unicode__(self):
        return self.name

    def get_admin_template(self):
        return "admin/homepage_blocks/partials/%s.html" % (self.slug)

    def get_public_template(self):
        return "public/blocks/%s.html" % (self.slug)

class HomepageSettings(models.Model):

    banner_strapline = models.CharField(max_length=200,blank=True,null=True)
    banner = models.ImageField(upload_to='homepage_banner',blank=True,null=True)
    mobile_banner = models.ImageField(upload_to='homepage_banner',blank=True,null=True)

    main_title = models.CharField(max_length=200,blank=True,null=True)
    main_subtitle = models.TextField(blank=True,null=True)

    secondary_title = models.CharField(max_length=200,blank=True,null=True)
    left_content = models.TextField(blank=True)
    right_content = models.TextField(blank=True)

    block_layout = models.ForeignKey('BlockLayout',blank=True,null=True)

    def __unicode__(self):
        return "Homepage Settings"

    def clean(self):
        validate_only_one_instance(self)

class HomepageBlock(models.Model):

    TYPE_CHOICES = (
        (u'normal',u'Normal'),
        (u'banner',u'Banner'),
        (u'events-feed','Events Feed'),
        (u'news-feed','News Feed'),
    )

    name        = models.CharField(max_length=150)
    content     = models.TextField(blank=True,null=True)
    image       = models.ImageField(blank=True,null=True,upload_to='widgets')
    link        = models.CharField(max_length=150,blank=True,null=True)
    block_type = models.CharField(max_length=20,blank=True,null=True,choices=TYPE_CHOICES)

    location    = models.CharField(max_length=20,blank=True,null=True)
    block_layout = models.ForeignKey('BlockLayout',blank=True,null=True)
    order        = models.IntegerField(default=0)

    active      = models.BooleanField(default=False)
    sidebar     = models.BooleanField(default=False,verbose_name='Show Block in Sidebar')

    def __unicode__(self):
        return self.name

    def get_images(self):
        images = BannerImage.objects.filter(block=self)
        return images

    def get_template(self):
        if self.block_type == 'banner':
            return 'public/blocks/banner.html'
        elif self.block_type == 'news-feed':
            return 'public/blocks/news-feed.html'
        elif self.block_type == 'events-feed':
            return 'public/blocks/events-feed.html'
        else:
            return 'public/blocks/block.html'

    def get_vars(self):

        if self.block_type == 'banner':
            args = {'images':self.get_images()}
        elif self.block_type == 'news-feed':
            args = self.get_news_feed()
        elif self.block_type == 'events-feed':
            args = self.get_events_feed()
        else:
            args = False

        return args

    def get_news_feed(self):
        from modules.mediacentre.models import *
        try:
            today = date.today()
            news_cat = MediaSection.objects.get(slug='news')
            articles = MediaArticle.objects.filter(section=news_cat, published=True,archive_date__gt=today).order_by('-sticky',news_cat.order)[:2]

            args = {'news_cat':news_cat,'articles':articles}
            return args

        except:
            return False

    def get_events_feed(self):
        from modules.mediacentre.models import *
        try:
            today = date.today()
            events_category = MediaSection.objects.get(slug='events')
            articles = MediaArticle.objects.filter(section=events_category, published=True,event_date_end__gt=today).order_by('-sticky',events_category.order)

            args = {'events_cat':events_category,'articles':articles}
            return args

        except:
            return False

class BannerImage(models.Model):

    block = models.ForeignKey('HomepageBlock')
    order = models.IntegerField()

    title = models.CharField(max_length=200)
    link = models.CharField(max_length=100)
    sub_title = models.CharField(max_length=200,blank=True,null=True)
    image = models.ImageField(upload_to='widgets/banners')

    def __unicode__(self):
        return "%s - %s" % (self.block, self.title)



def validate_only_one_instance(obj):
    model = obj.__class__
    if (model.objects.count() > 0 and
            obj.id != model.objects.get().id):
        raise ValidationError("Can only create 1 %s instance" % model.__name__)


class ContentBlock(models.Model):

    name = models.CharField(max_length=200)
    slug = models.SlugField(max_length=200)
    description = models.TextField()
    content = models.TextField()

    def __unicode__(self):
        return self.name

class HomepageSearch(models.Model):

    TYPE_CHOICES = (
        (u'content-type',u'Content Type'),
        (u'audience',u'Audience'),
    )

    name = models.CharField(max_length=200)
    slug = models.SlugField(max_length=200)
    type = models.CharField(max_length=200,choices=TYPE_CHOICES)
    order = models.IntegerField()

    def __unicode__(self):
        return self.name
