from django.db import models
from datetime import datetime, date, time

from modules.members.models import *

class MediaSection(models.Model):

    ORDER_CHOICES = (
        (u'created', u'Oldest First'),
        (u'-created', u'Newest First'),
        (u'title', u'Alphabetical'),
        (u'-title', u'Alphabetical Reverse'),
        (u'event_date_start','Nearest Event First'),
        (u'-event_date_start','Furthest Away Event First'),
    )

    TYPE_CHOICES = (
        (u'news', u'News'),
        (u'events', u'Events'),
    )

    name    = models.CharField(max_length=150)
    slug    = models.SlugField(unique=True,max_length=160)
    type    = models.CharField(max_length=50, choices=TYPE_CHOICES)
    order   = models.CharField(max_length=50, choices=ORDER_CHOICES)

    def __unicode__(self):
        return self.name

    def get_articles(self):
        articles = MediaArticle.objects.filter(section=self, published=True)
        return articles

    def get_current_articles(self):
        articles = MediaArticle.objects.filter(section=self,published=True).order_by(self.order)
        if self.type == 'events':
            today = date.today()
            articles = articles.filter(event_date_start__gte=today)
        return articles

    def get_latest_articles(self):
        today = datetime.now()
        articles = MediaArticle.objects.filter(section=self, published=True).order_by(self.order)[:2]
        return articles

class MediaArticle(models.Model):

    section             = models.ForeignKey('MediaSection')
    title               = models.CharField(max_length=150)
    slug                = models.SlugField(unique=True,max_length=160)
    description         = models.TextField(blank=True, verbose_name='Snippet')
    keywords            = models.TextField(blank=True,help_text='These are used for search, please enter comma separated.')
    content             = models.TextField()
    created             = models.DateTimeField(auto_now_add=True)
    archive_date		= models.DateField(blank=True,null=True)
    published           = models.BooleanField(default=True)
    sticky              = models.BooleanField(default=False,verbose_name='Pull article to top of list')
    event_date_start    = models.DateField(blank=True, null=True,verbose_name='Date Event Starts')
    event_date_end      = models.DateField(blank=True, null=True,verbose_name='Date Event Ends')
    limit_to_member_types   = models.ManyToManyField('members.MemberType',blank=True)
    limit_to_member_groups  = models.ManyToManyField('members.MemberGroup',blank=True)

    search_terms            = models.ManyToManyField('blocks.HomepageSearch',blank=True)

    def __unicode__(self):
        return self.title

    def get_resources(self):
        resources = MediaArticleResource.objects.filter(article=self)
        return resources

    def get_permission(self,request):

        if request.user.is_authenticated():
            if request.user.is_superuser:
                return 'allowed'

            try:
                member = Member.objects.get(user=request.user)
                if member.is_current():
                    if self.limit_to_member_types.all():
                        if not member.member_type in self.limit_to_member_types.all():
                            return 'permission_denied'
                    if self.limit_to_member_groups.all():
                        in_groups = False
                        for group in self.limit_to_member_groups.all():
                            if member in group.members.all():
                                in_groups = True

                        if not in_groups:
                            return 'permission-denied'
                else:
                    if self.limit_to_member_types.all() or self.limit_to_member_groups.all():
                        return 'denied'
            except Member.DoesNotExist:
                if self.limit_to_member_groups.all() or self.limit_to_member_types.all():
                    return 'denied'

        else:
            if self.limit_to_member_groups.all() or self.limit_to_member_types.all():
                return 'denied'

        return 'allowed'


class MediaArticleResource(models.Model):

    name = models.CharField(max_length=150)
    article = models.ForeignKey(MediaArticle)
    resource = models.ForeignKey('resources.Resource')
    order = models.IntegerField()

    def __unicode__(self):
        return self.name
