from django.db import models
from django.contrib.auth.models import Group

from modules.members.models import Member
from modules.file_manager.fields import *

class Page(models.Model):

    title                   = models.CharField(max_length=150)
    url                     = models.SlugField(max_length=150,unique=True,blank=True,null=True)
    redirect_to             = models.CharField(max_length=200,blank=True,null=True)
    parent                  = models.ForeignKey('self', blank=True, null=True)
    order                   = models.IntegerField(blank=True,null=True)
    intro_page              = models.BooleanField(default=False,verbose_name='Is an Intro Page')
    introduction            = models.TextField(blank=True)
    keywords                = models.TextField(blank=True,help_text='These are used for search, please enter comma separated.')
    introduction_image      = models.ImageField(upload_to='pages',blank=True,null=True)
    content                 = models.TextField(blank=True)
    limit_to_member_types   = models.ManyToManyField('members.MemberType',blank=True)
    limit_to_member_groups  = models.ManyToManyField('members.MemberGroup',blank=True)

    search_terms            = models.ManyToManyField('blocks.HomepageSearch',blank=True)

    #cms permissions
    groups_allowed  = models.ManyToManyField(Group,blank=True,verbose_name='Groups Allowed to Edit')

    def get_full_url(self):
        breadcrumbs = self.get_breadcrumbs()
        url = ''
        for breadcrumb in breadcrumbs:
            url = "%s%s/" % (url,breadcrumb.url)
        return url

    def get_subpages_admin(self):
        return Page.objects.filter(parent=self).order_by('order')

    def get_subpages(self):
        pages = Page.objects.filter(parent=self).order_by('order')
        return pages

    def get_breadcrumbs(self):
        breadcrumbs = []
        breadcrumbs.append(self)

        if self.parent:
            parent_breadcrumbs = self.parent.get_parent_breadcrumbs(breadcrumbs)
            breadcrumbs.reverse()
            return breadcrumbs
        else:
            return breadcrumbs

    def get_parent_breadcrumbs(self,breadcrumbs):

        breadcrumbs.append(self)

        if self.parent:
            breadcrumbs = self.parent.get_parent_breadcrumbs(breadcrumbs)
            return breadcrumbs
        else:
            return breadcrumbs

    def get_top_level(self):
        if self.parent:
            top_level = self.parent.get_top_level()
            return top_level
        else:
            return self

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ['title']

    def get_sections(self):
        return PageSection.objects.filter(page=self).order_by('order')

    def get_resources(self):
        return PageResource.objects.filter(page=self).order_by('order')

    def get_blocks(self):
        return PageBlock.objects.filter(page=self).order_by('order')

    def get_permission(self,request):

        if request.user.is_authenticated():
            if request.user.is_superuser:
                return 'allowed'

            try:
                member = Member.objects.get(user=request.user)
                if member.is_current():
                    if self.limit_to_member_types.all():
                        if not member.member_type in self.limit_to_member_types.all():
                            return 'permission_denied'
                    if self.limit_to_member_groups.all():
                        in_groups = False
                        for group in self.limit_to_member_groups.all():
                            if member in group.members.all():
                                in_groups = True

                        if not in_groups:
                            return 'permission-denied'
                else:
                    if self.limit_to_member_types.all() or self.limit_to_member_groups.all():
                        return 'denied'
            except Member.DoesNotExist:
                if self.limit_to_member_groups.all() or self.limit_to_member_types.all():
                    return 'denied'
        else:
            if self.limit_to_member_groups.all() or self.limit_to_member_types.all():
                return 'denied'

        return 'allowed'


    def can_edit(self,user):

        if user.has_perm('admin_users.can_access_pages'):
            return True
        elif user.has_perm('admin_users.can_edit_certain_pages'):
            if self.groups_allowed.all():
                allowed = False
                for group in self.groups_allowed.all():
                    if group in user.groups.all():
                        allowed = True
                return allowed
            else:
                return False
        else:
            return False


class PageResource(models.Model):

    name = models.CharField(max_length=150)
    page = models.ForeignKey(Page)
    resource = models.ForeignKey('resources.Resource')
    order = models.IntegerField()

    def __unicode__(self):
        return self.name

class PageSection(models.Model):
    title = models.CharField(max_length=150)
    order = models.IntegerField()
    content = models.TextField(blank=True)
    page = models.ForeignKey(Page)
    is_title = models.BooleanField(default=False)

    def __unicode__(self):
        return self.title

    def get_resources(self):
        return PageSectionResource.objects.filter(section=self).order_by('order')


class PageBlock(models.Model):

    title = models.CharField(max_length=200)
    content = models.TextField()
    link = models.CharField(max_length=200)
    image = ImageUploadField('file_manager.Image')

    page = models.ForeignKey(Page)
    order = models.IntegerField()

    def __unicode__(self):
        return self.title

class PageSectionResource(models.Model):

    name = models.CharField(max_length=150)
    section = models.ForeignKey(PageSection)
    resource = models.ForeignKey('resources.Resource')
    order = models.IntegerField()

    def __unicode__(self):
        return self.name
