from django.db import models

from django.conf import settings

from django.core.files.storage import FileSystemStorage

fs = FileSystemStorage(location=settings.PRIVATE_FILES)

class PaperCategory(models.Model):
    
    name = models.CharField(max_length=100)
    order = models.IntegerField(default=0)
    
    def __unicode__(self):
        return self.name
        
    def get_papers(self):
        papers = Paper.objects.filter(category=self,enabled=True)
        return papers

    
class Paper(models.Model):
    
    name = models.CharField(max_length=100)
    slug = models.SlugField(unique=True)
    category = models.ForeignKey('PaperCategory')
    
    subtitle = models.TextField(blank=True,null=True)
    
    content_left = models.TextField(blank=True)
    content_right = models.TextField(blank=True)
    
    pdf_file = models.FileField(upload_to='papers',storage=fs,blank=True)
    external_url = models.CharField(max_length=200,blank=True,null=True,help_text='Paste the external url above')
    embed_code = models.TextField(blank=True,help_text='Paste your embed code above')
    
    cpd_credits = models.IntegerField(default=0)
    pass_percentage = models.IntegerField(default=0)
    
    enabled = models.BooleanField(default=False)
    
    def __unicode__(self):
        return self.name
    
    def get_sections(self):
        paper_sections = PaperSection.objects.filter(paper=self).order_by('order')
        return paper_sections
        
    def get_all_questions(self):
        
        questions = PaperQuestion.objects.filter(paper=self)
        return questions
        
    def get_questions(self):
        
        questions = PaperQuestion.objects.filter(paper=self,paper_section__isnull=True).order_by('order')
        return questions
        
    def get_next_section_order(self):
        
        sections = self.get_sections()
        if sections:
            sections.reverse()
            section = sections[0]
            order = section.order + 1
            return order
        else:
            return 1
            
    def get_next_order(self):
        
        questions = self.get_questions()
        
        if questions:
            questions = questions.reverse()
                        
            question = questions[0]
            order = question.order + 1
            return order
        else:
            return 1

class PaperSection(models.Model):
    
    paper = models.ForeignKey('Paper')
    name = models.CharField(max_length=200)
    order = models.IntegerField()
    
    def __unicode__(self):
        return self.name
        
    class Meta:
        ordering = ['order']
        
    def get_questions(self):
        questions = PaperQuestion.objects.filter(paper_section=self).order_by('order')
        return questions
        
    def get_next_order(self):
        
        questions = self.get_questions()
        if questions:
            questions.reverse()
            question = questions[0]
            order = question.order + 1
            return order
        else:
            return 1
        

class PaperQuestion(models.Model):
    
    TYPE_OPTIONS = (
        (u'select','Selection (Choose One)'),
        (u'multi-select','Selection (Choose Multiple)'),
        (u'true-false','True or False')
    )
    
    """(u'text-box','Single Line (Text Box)'),
        (u'text-area','Multiple Line (Text Area)'),"""
    
    
    paper = models.ForeignKey('Paper')
    paper_section = models.ForeignKey('PaperSection',blank=True,null=True)
    
    name = models.TextField(max_length=200,verbose_name='Question')
    type = models.CharField(max_length=200,choices=TYPE_OPTIONS)
    required = models.BooleanField(default=True)
    order = models.IntegerField()
    
    options = models.TextField(blank=True,null=True)
    answer = models.TextField(blank=True,null=True)
    
    def __unicode__(self):
        return self.name
        
    def get_options(self):
        
        if self.options:
            options = [x.strip() for x in self.options.split(',')]
            return options
        else:
            return False
        
    class Meta:
        ordering = ['paper_section__order','order']
        

class PaperSubmission(models.Model):
    
    paper = models.ForeignKey('Paper')
    member = models.ForeignKey('members.Member')
    submitted = models.DateTimeField(blank=True,null=True)
    complete = models.BooleanField(default=False)
    attempt = models.IntegerField(default=0)
    last_saved = models.DateTimeField(auto_now=True)
    
    certificate = models.FileField(upload_to='cetificates',storage=fs,blank=True,null=True)
    
    def __unicode__(self):
        return "%s - %s" % (self.paper,self.member)
        
    class Meta:
        ordering = ['-last_saved']
        
        
    def get_answers(self):
        answers = PaperQuestionSubmission.objects.filter(submission=self)
        return answers
        
    def get_incorrect_answers(self):
        answers = PaperQuestionSubmission.objects.filter(submission=self,correct=False)
        return answers
        
    def get_incorrect_questions(self):
        answers = self.get_incorrect_answers()
        questions = []
        for answer in answers:
            questions.append(answer.question)
        
        return questions
        
    def get_total(self):
        
        count = 0
        answers = self.get_answers()
        for answer in answers:
            if answer.correct:
                count = count + 1
        return count
        
    def get_max_score(self):
        
        questions = self.paper.get_all_questions()
        count = 0
        for question in questions:
            count = count + 1
            
        return count
        
    def get_percentage(self):
        
        if self.get_total() > 0 and self.get_max_score() > 0:
            
            total  = float(self.get_total())
            max_score = float(self.get_max_score())
                        
            percent = (total / max_score) * 100
                        
            return percent
            
        else:
            return 0
        
class PaperQuestionSubmission(models.Model):
    
    submission = models.ForeignKey('PaperSubmission')
    question = models.ForeignKey('PaperQuestion',blank=True,null=True,on_delete=models.SET_NULL)
    question_name = models.TextField()
    answer = models.TextField(blank=True)
    correct = models.BooleanField(default=False)
    order = models.IntegerField()
    
    def __unicode__(self):
        return "%s - %s" % (self.submission,self.question)
        
    def get_answers(self):
        
        if self.question.type == 'multi-select':        
            if self.answer:
                answers = [x.strip() for x in self.answer.split(',')]
                return answers
            else:
                return False
        else:
            return False