#Django core bits
from django.shortcuts import render_to_response, get_object_or_404
from django.template import RequestContext, loader, Context
from django.contrib.auth.decorators import login_required, permission_required
from django.db.models import Q
from django.contrib.sites.models import Site
from django.conf import settings
from django.http import HttpResponseRedirect, Http404, HttpResponse
from django.core.urlresolvers import reverse
from django.contrib import messages
from django.contrib.auth.models import User, Group
from django.template.defaultfilters import slugify

#models
from models import *

#forms
from forms import *

#functions
from modules.core.functions import *
from modules.core.decorators import *

#Others
import random,csv

#admin views

@permission_required('admin_users.can_access_resources')
def admin_resources(request):
    
    categories = ResourceCategory.objects.all().order_by('name')

    return render_to_response('admin/resources/admin-resources.html',{'categories':categories},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_add_resource_category(request):
    
    form = ResourceCategoryForm()
    
    if request.POST:
        form = ResourceCategoryForm(request.POST)
        if form.is_valid():
            category = form.save(commit=False)
            category.slug = slugify_unique(category.name, ResourceCategory)
            category.save()
            
            messages.success(request,'Resource Category Added')
            return HttpResponseRedirect(reverse('admin_resources'))
    return render_to_response('admin/resources/add-resource-category.html',{'form':form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_edit_resource_category(request,category_id):
    
    category = get_object_or_404(ResourceCategory,id=category_id)
    
    form = ResourceCategoryForm(instance=category)
    
    if request.POST:
        form  = ResourceCategoryForm(request.POST,instance=category)
        if form.is_valid():
            form.save()
            messages.success(request,'Resource Category Updated')
            return HttpResponseRedirect(reverse('admin_resources'))
            
    return render_to_response('admin/resources/edit-resource-category.html',{'category':category,'form':form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_delete_resource_category(request,category_id):

    category = get_object_or_404(ResourceCategory,id=category_id)

    try:
        category.delete()
        messages.success(request,'Resource Category Deleted')
    except:
        messages.error(request,'Could not delete category')
    return HttpResponseRedirect(reverse('admin_resources'))

@permission_required('admin_users.can_access_resources')
def admin_category_resources(request,category_id):
	
    category = get_object_or_404(ResourceCategory,id=category_id)
    resources = Resource.objects.filter(category=category).order_by('name')
	
    return render_to_response('admin/resources/admin-category-resources.html',{'category':category,'resources':resources},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_add_resource(request,category_id):
    
    category = get_object_or_404(ResourceCategory,id=category_id)
    additional_fields = ResourceAdditionalField.objects.filter(category=category)
    
    form = ResourceForm()

    if request.POST:
        form = ResourceForm(request.POST,request.FILES)
    	if form.is_valid():
            resource = form.save(commit=False)
            resource.slug = slugify_unique(resource.name, Resource)
            resource.category = category
            resource.save()
            form.save_m2m()
            
            for field in additional_fields:
                if field.type == 'text':
                    if request.POST.get(field.slug):
                        field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,value=request.POST.get(field.slug))
                        field_value.save()
                elif field.type == 'file' or field.type == 'image':
                    if request.FILES.get(field.slug):
                        field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,file_value=request.FILES.get(field.slug))
                        field_value.save()
                elif field.type == 'checkbox':
                    if request.POST.get(field.slug):
                        field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,boolean_value=True)
                        field_value.save()
                    else:
                        field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,boolean_value=False)
                        field_value.save()

            messages.success(request, 'Resource Added')
            return HttpResponseRedirect(reverse('admin_category_resources',args=[category.id]))
    
    return render_to_response('admin/resources/add-resource.html', {'form': form, 'category':category,'additional_fields':additional_fields}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_edit_resource(request,category_id,resource_id):

    category = get_object_or_404(ResourceCategory,id=category_id)

    resource = get_object_or_404(Resource,id=resource_id)
    
    additional_fields = ResourceAdditionalField.objects.filter(category=category)
    additional_field_values = ResourceAdditionalFieldValue.objects.filter(resource=resource)

    form = ResourceForm(instance=resource)

    if request.POST:
        form = ResourceForm(request.POST,request.FILES,instance=resource)

        if form.is_valid():
            form.save()
            
            for field in additional_fields:
                if field.type == 'text':
                    if request.POST.get(field.slug):
                        try:
                            value = ResourceAdditionalFieldValue.objects.get(resource=resource,additional_field=field)
                            value.value = request.POST.get(field.slug)
                            value.save()
                        except ResourceAdditionalFieldValue.DoesNotExist:    
                            field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,value=request.POST.get(field.slug))
                            field_value.save()
                elif field.type == 'file' or field.type == 'image':
                    if request.FILES.get(field.slug):
                        try:
                            value = ResourceAdditionalFieldValue.objects.get(resource=resource,additional_field=field)
                            value.file_value = request.FILES.get(field.slug)
                            value.save()
                        except ResourceAdditionalFieldValue.DoesNotExist:
                            field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,file_value=request.FILES.get(field.slug))
                            field_value.save()
                elif field.type == 'checkbox':
                    if request.POST.get(field.slug):
                        try:
                            value = ResourceAdditionalFieldValue.objects.get(resource=resource,additional_field=field)
                            value.boolean_value = True
                            value.save()
                        except ResourceAdditionalFieldValue.DoesNotExist:
                            field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,boolean_value=True)
                            field_value.save()
                    else:
                        try:
                            value = ResourceAdditionalFieldValue.objects.get(resource=resource,additional_field=field)
                            value.boolean_value = False
                            value.save()
                        except ResourceAdditionalFieldValue.DoesNotExist:
                            field_value = ResourceAdditionalFieldValue(resource=resource,additional_field=field,boolean_value=False)
                            field_value.save()

            messages.success(request, 'Resource Updated')
            
            return HttpResponseRedirect(reverse('admin_category_resources',args=[category.id]))
	
    return render_to_response('admin/resources/edit-resource.html', {'form': form, 'category':category,'resource':resource,'additional_fields':additional_fields,'additional_field_values':additional_field_values}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_delete_resource(request,category_id,resource_id):

    category = get_object_or_404(ResourceCategory,id=category_id)
    
    resource = get_object_or_404(Resource,id=resource_id)

    try:
        resource.delete()
        messages.success(request, 'Resource Deleted')
    except:
        messages.error(request, 'Could not delete resource')
    return HttpResponseRedirect(reverse('admin_category_resources',args=[category.id]))
    
    
@permission_required('admin_users.can_access_resources')
def admin_category_additional_fields(request,category_id):
	
    category = get_object_or_404(ResourceCategory,id=category_id)
    additional_fields = ResourceAdditionalField.objects.filter(category=category).order_by('order')
	
    return render_to_response('admin/resources/admin-category-additional-fields.html',{'category':category,'additional_fields':additional_fields},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_add_additional_field(request,category_id):
    
    category = get_object_or_404(ResourceCategory,id=category_id)
    
    form = AdditionalFieldForm()
    
    if request.POST:
        form = AdditionalFieldForm(request.POST)
    	if form.is_valid():
            field = form.save(commit=False)
            field.slug = slugify(field.name)
            field.category = category
            field.save()
            messages.success(request, 'Additional Field Added')
            return HttpResponseRedirect(reverse('admin_category_additional_fields',args=[category.id]))
    
    return render_to_response('admin/resources/add-additional-field.html', {'form': form, 'category':category}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_edit_additional_field(request,category_id,field_id):

    category = get_object_or_404(ResourceCategory,id=category_id)

    field = get_object_or_404(ResourceAdditionalField,id=field_id)

    form = AdditionalFieldForm(instance=field)

    if request.POST:
        form = AdditionalFieldForm(request.POST,instance=field)

        if form.is_valid():
            form.save()
            messages.success(request, 'Additional Field Updated')
            return HttpResponseRedirect(reverse('admin_category_additional_fields',args=[category.id]))
	
    return render_to_response('admin/resources/edit-additional-field.html', {'form': form, 'category':category,'field':field}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_resources')
def admin_delete_additional_field(request,category_id,field_id):

    category = get_object_or_404(ResourceCategory,id=category_id)
    
    resource = get_object_or_404(ResourceAdditionalField,id=field_id)

    try:
        resource.delete()
        messages.success(request, 'Field Deleted')
    except:
        messages.error(request, 'Could not delete field')
    return HttpResponseRedirect(reverse('admin_category_additional_fields',args=[category.id]))