#Django core bits
from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, permission_required
from django.http import HttpResponseRedirect, Http404, HttpResponse
from django.core.urlresolvers import reverse
from django.contrib import messages
from django.template.defaultfilters import slugify
import csv

from modules.core.functions import *

#Forms
from forms import *

#Models
from models import *

@permission_required('admin_users.can_access_travel_grants')
def tg_forms(request):

    enabled_tg_forms = TGForm.objects.filter(enabled=True)
    disabled_tg_forms = TGForm.objects.filter(enabled=False)

    return render(request, 'admin/travel_grants/tg-forms.html', {'enabled_tg_forms':enabled_tg_forms, 'disabled_tg_forms':disabled_tg_forms})


@permission_required('admin_users.can_access_travel_grants')
def add_tgform(request):

    tg_form_form = TGFormForm()

    if request.POST:

        tg_form_form = TGFormForm(request.POST)
        if tg_form_form.is_valid():
            tg_form = tg_form_form.save(commit=False)
            tg_form.slug = slugify_unique(tg_form.name, TGForm, 'slug')
            tg_form.save()

            messages.success(request, 'Travel Grant Form Created')

            return redirect(reverse('admin_travel_grants'))

    return render(request, 'admin/travel_grants/add-tg-form.html', {'form':tg_form_form})

@permission_required('admin_users.can_access_travel_grants')
def edit_tgform(request, form_id):

    tg_form = get_object_or_404(TGForm, id=form_id)
    tg_form_form = TGFormForm(instance=tg_form)

    if request.POST:

        tg_form_form = TGFormForm(request.POST, instance=tg_form)
        if tg_form_form.is_valid():
            tg_form_form.save()

            messages.success(request, 'Travel Grant Form Updated')

            return redirect(reverse('admin_travel_grants'))

    return render(request, 'admin/travel_grants/edit-tg-form.html', {'tg_form':tg_form, 'form':tg_form_form})


@permission_required('admin_users.can_access_travel_grants')
def delete_tgform(request, form_id):

    tg_form = get_object_or_404(TGForm, id=form_id)
    tg_form.delete()

    messages.success(request, 'TG Form Deleted')

    return redirect(reverse('admin_travel_grants'))


@permission_required('admin_users.can_access_travel_grants')
def applications(request, form_id):

    tg_form = get_object_or_404(TGForm, id=form_id)
    applications = TGApplication.objects.filter(form=tg_form)

    return render(request, 'admin/travel_grants/applications.html', {'tg_form':tg_form, 'applications':applications})


@permission_required('admin_users.can_access_travel_grants')
def download_applications(request, form_id):

    tg_form = get_object_or_404(TGForm, id=form_id)
    applications = TGApplication.objects.filter(form=tg_form)

    fields = ['Title', 'Forename', 'Surname', 'Email', 'Telephone', 'Institution', 'Address', 'Abstract', 'Intended Date', 'Objective', 'Declaration', 'Contributors']
    
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=travel_grant_applications.csv"

    writer = csv.writer(response)
    writer.writerow(fields)

    for application in applications:

        fields = [application.title, application.forename, application.surname, application.email, application.telephone, application.institution, application.address, application.abstract, application.intended_date, application.objective, application.confirm_declaration]
        for contributor in application.get_contributors():
            fields += [contributor.title, contributor.forename, contributor.surname, contributor.age, contributor.post_held]

        writer.writerow(fields)
    
    return response


@permission_required('admin_users.can_access_travel_grants')
def view_application(request, form_id, application_id):

    tg_form = get_object_or_404(TGForm, id=form_id)
    application = get_object_or_404(TGApplication, id=application_id)

    return render(request, 'admin/travel_grants/view-application.html', {'tg_form':tg_form, 'application':application})


@permission_required('admin_users.can_access_travel_grants')
def delete_application(request, form_id, application_id):

    tg_form = get_object_or_404(TGForm, id=form_id)
    application = get_object_or_404(TGApplication, id=application_id)

    application.delete()

    messages.success(request, 'Application has been deleted')

    return redirect(reverse('admin_tg_applications', args=[tg_form.id]))