from django.db import models

from signals import *
from modules.notifications.models import *

class Evaluation(models.Model):

    name = models.CharField(max_length=200)
    slug = models.SlugField()
    description = models.TextField()

    enabled = models.BooleanField(default=False)

    event_name = models.CharField(max_length=200,blank=True,null=True)
    event_date = models.CharField(max_length=200,blank=True,null=True)
    event_venue = models.TextField(blank=True,null=True)
    event_author = models.TextField(blank=True,null=True)
    event_signature = models.ImageField(upload_to='evaluations')

    def __unicode__(self):
        return self.name

    def get_submissions(self):
        submissions = EvaluationSubmission.objects.filter(evaluation=self)
        return submissions

    def get_days(self):
        days = EvaluationDay.objects.filter(evaluation=self)
        return days

    def get_total_credits(self):

        days = self.get_days()
        cpd_points = 0
        for day in days:
            cpd_points = cpd_points + day.cpd_points
        return cpd_points

    def get_questions(self):
        questions = EvaluationQuestion.objects.filter(evaluation=self)
        return questions

    def get_areas(self):
        areas = EvaluationArea.objects.filter(evaluation=self)
        return areas

class EvaluationArea(models.Model):

    SCALE_CHOICES  = (
        (u'poor-excellent',u'Poor to Excellent'),
        (u'numeric',u'1 to 5'),
        (u'more-less',u'More,Less')
    )

    evaluation = models.ForeignKey(Evaluation)
    name = models.CharField(max_length=200)
    order = models.IntegerField()
    scale = models.CharField(max_length=20,choices=SCALE_CHOICES)

    def __unicode__(self):
        return "%s  - %s" % (self.evaluation.name,self.name)

    def get_items(self):
        items = EvaluationItem.objects.filter(evaluation_area=self)
        return items

class EvaluationItem(models.Model):

    evaluation_area = models.ForeignKey(EvaluationArea)
    name = models.CharField(max_length=200)
    order = models.IntegerField()
    did_not_attend_option = models.BooleanField(default=False, verbose_name='Show Did Not Attend Option')

    def __unicode__(self):
        return "%s - %s - %s" % (self.evaluation_area.evaluation.name,self.evaluation_area.name,self.name)


class EvaluationQuestion(models.Model):

    TYPE_CHOICES = (
        (u'text',u'Text Box'),
        (u'text-area',u'Text Area'),
    )

    evaluation = models.ForeignKey(Evaluation)
    name = models.CharField(max_length=200)
    type = models.CharField(max_length=200,choices=TYPE_CHOICES)
    required = models.BooleanField(default=False)

    order = models.IntegerField()

    def __unicode__(self):
        return "%s - %s" % (self.evaluation.name,self.name)

class EvaluationDay(models.Model):

    evaluation = models.ForeignKey(Evaluation)
    name = models.CharField(max_length=200)
    cpd_points = models.IntegerField()
    order = models.IntegerField()

    def __unicode__(self):
        return "%s - %s" % (self.evaluation.name,self.name)


class EvaluationSubmission(models.Model):

    evaluation      = models.ForeignKey(Evaluation)
    evaluation_name = models.CharField(max_length=200)

    website_user    = models.ForeignKey('members.Member',blank=True,null=True)
    name            = models.CharField(max_length=200)
    email_address   = models.EmailField(max_length=200)

    created = models.DateTimeField(max_length=200,auto_now_add=True)
    certificate = models.FileField(upload_to='members/certificates',blank=True,null=True)

    def __unicode__(self):
        return "%s - %s" %  (self.name,self.evaluation_name)

    def get_item_submissions(self):
        items  = EvaluationItemSubmission.objects.filter(evaluation_submission=self).order_by('evaluation_item__evaluation_area__order','evaluation_item__order')
        return items

    def get_question_submissions(self):
        questions = EvaluationQuestionSubmission.objects.filter(evaluation_submission=self).order_by('evaluation_question__order')
        return questions

    def get_day_submissions(self):
        days = EvaluationDaySubmission.objects.filter(evaluation_submission=self).order_by('evaluation_day__order')
        return days

    def get_total_credits(self):

        days = self.get_day_submissions()
        total = 0
        for day in days:
            total = total + day.evaluation_day_points
        return total

class EvaluationItemSubmission(models.Model):

    evaluation_submission = models.ForeignKey(EvaluationSubmission)

    evaluation_item = models.ForeignKey(EvaluationItem,blank=True,null=True,on_delete=models.SET_NULL)
    item_name = models.CharField(max_length=200)
    choice = models.CharField(max_length=200)

    def __unicode__(self):
        return "%s - %s - %s" % (self.evaluation_submission.name,self.evaluation_submission.evaluation_name,self.item_name)

class EvaluationQuestionSubmission(models.Model):

    evaluation_submission = models.ForeignKey(EvaluationSubmission)

    evaluation_question = models.ForeignKey(EvaluationQuestion,blank=True,null=True,on_delete=models.SET_NULL)
    question_name = models.CharField(max_length=200)
    answer = models.TextField()

    def __unicode__(self):
        return "%s - %s - %s" % (self.evaluation_submission.name,self.evaluation_submission.evaluation_name,self.question_name)

class EvaluationDaySubmission(models.Model):

    evaluation_submission = models.ForeignKey(EvaluationSubmission)

    evaluation_day = models.ForeignKey(EvaluationDay,blank=True,null=True,on_delete=models.SET_NULL)
    evaluation_day_name = models.CharField(max_length=200)
    evaluation_day_points = models.IntegerField()

    def __unicode__(self):
        return "%s - %s - %s" % (self.evaluation_submission.name,self.evaluation_submission.evaluation_name,self.evaluation_day_name)
