#Django core bits
from django.shortcuts import render_to_response, get_object_or_404
from django.template import RequestContext, loader, Context
from django.contrib.auth.decorators import login_required, permission_required
from django.db.models import Q
from django.contrib.sites.models import Site
from django.conf import settings
from django.http import HttpResponseRedirect, Http404, HttpResponse
from django.core.urlresolvers import reverse
from django.contrib.auth.models import User, Group
from django.contrib import messages
from django.template.defaultfilters import slugify
from datetime import datetime, date, time


#Forms
from forms import *

#Models
from models import *

from modules.core.functions import *
from modules.core.decorators import *

@permission_required('admin_users.can_access_mediacentre')
def admin_mediacentre(request):

    sections = MediaSection.objects.all().order_by('name')

    return render_to_response('admin/mediacentre/mediacentre.html', {'sections':sections}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_mediacentre')
def admin_edit_section(request, section_slug):

    section = get_object_or_404(MediaSection, slug=section_slug)
    form = MediaSectionForm(instance=section)

    if request.POST:
        form = MediaSectionForm(request.POST, instance=section)
        if form.is_valid():
            try:
                form.save()
                messages.success(request, 'Media Centre Section Updated')
            except:
                messages.error(request, 'Could not Update Media Centre Section')

            return HttpResponseRedirect(reverse('admin_mediacentre'))

    return render_to_response('admin/mediacentre/edit_section.html', {'section':section, 'form': form}, context_instance=RequestContext(request))


@permission_required('admin_users.can_access_mediacentre')
def admin_articles(request, section_slug):

    section = get_object_or_404(MediaSection, slug=section_slug)
    articles = MediaArticle.objects.filter(section=section).order_by('-created')

    return render_to_response('admin/mediacentre/articles.html', {'section':section, 'articles': articles}, context_instance=RequestContext(request))


@permission_required('admin_users.can_access_mediacentre')
def admin_add_article(request, section_slug):

    section = get_object_or_404(MediaSection, slug=section_slug)

    if section.type == 'events':
        form = MediaArticleEventForm()
    else:
        form = MediaArticleForm()

    if request.POST:
        if section.type == 'events':
            form = MediaArticleEventForm(request.POST,request.FILES)
        else:
            form = MediaArticleForm(request.POST,request.FILES)
        if form.is_valid():
            #try:
            article = form.save(commit=False)
            article.section = section
            article.slug = slugify_unique(article.title, MediaArticle)
            article.save()
            form.save_m2m()
            messages.success(request, 'Article Created')
            #except:
            #    messages.error(request, 'Could not create article')
            
            return HttpResponseRedirect(reverse('admin_mediacentre_articles', args=[section.slug]))

    return render_to_response('admin/mediacentre/add_article.html', {'section':section, 'form': form}, context_instance=RequestContext(request))


@permission_required('admin_users.can_access_mediacentre')
def admin_edit_article(request, section_slug, article_slug):

    article = get_object_or_404(MediaArticle, slug=article_slug)

    if article.section.type == 'events':
        form = MediaArticleEventForm(instance=article)
    else:
        form = MediaArticleForm(instance=article)

    if request.POST:
        if article.section.type == 'events':
            form = MediaArticleEventForm(request.POST, request.FILES, instance=article)
        else:
            form = MediaArticleForm(request.POST, request.FILES, instance=article)
        if form.is_valid():
            try:
                form.save()
                messages.success(request, 'Article Updated')
            except:
                messages.error(request, 'Could not update article')
            return HttpResponseRedirect(reverse('admin_mediacentre_articles', args=[article.section.slug]))

    return render_to_response('admin/mediacentre/edit_article.html', {'section':article.section, 'form': form, 'article': article}, context_instance=RequestContext(request))


@permission_required('admin_users.can_access_mediacentre')
def admin_duplicate_article(request,section_slug,article_slug):
    
    section = get_object_or_404(MediaSection,slug=section_slug)
    article = get_object_or_404(MediaArticle, slug=article_slug)
    
    if section.slug != 'education-courses':
        raise Http404
    else:
        new_section = get_object_or_404(MediaSection,slug='junior-courses')
        try:
            new_article = MediaArticle(title=article.title,description=article.description,content=article.content,event_date_start=article.event_date_start,event_date_end=article.event_date_end)
            new_article.slug = slugify_unique(article.title, MediaArticle)
            new_article.section = new_section
            new_article.save()
            messages.success(request,'Article duplicated to Junior Courses')
        except:
            messages.error(request,'Could not duplicate article')
    return HttpResponseRedirect(reverse('admin_mediacentre_articles',args=[section.slug]))


@permission_required('admin_users.can_access_mediacentre')
def admin_delete_article(request, section_slug, article_slug):

    section = get_object_or_404(MediaSection,slug=section_slug)
    article = get_object_or_404(MediaArticle, slug=article_slug)

    try:
        article.delete()
        messages.success(request, 'Article Deleted')
    except:
        messages.error(request, 'Could not delete article')

    return HttpResponseRedirect(reverse('admin_mediacentre_articles',args=[section.slug]))


@permission_required('admin_users.can_access_mediacentre')
def admin_enable_article(request, section_slug, article_slug):

    section = get_object_or_404(MediaSection,slug=section_slug)
    article = get_object_or_404(MediaArticle, slug=article_slug)

    try:
        article.published = True
        article.save()
        messages.success(request, 'Article Published')
    except:
        messages.error(request, 'Could not publish article')
    return HttpResponseRedirect(reverse('admin_mediacentre_articles', args=[article.section.slug]))


@permission_required('admin_users.can_access_mediacentre')
def admin_disable_article(request, section_slug, article_slug):

    article = get_object_or_404(MediaArticle, slug=article_slug)

    try:
        article.published = False
        article.save()
        messages.success(request, 'Article Disabled')
    except:
        messages.error(request, 'Could not disable article')
    return HttpResponseRedirect(reverse('admin_mediacentre_articles', args=[article.section.slug]))
    
@permission_required('admin_users.can_access_mediacentre')
def admin_article_resources(request,section_slug,article_slug):

    article = get_object_or_404(MediaArticle, slug=article_slug)
    resources = MediaArticleResource.objects.filter(article=article).order_by('order')

    return render_to_response('admin/mediacentre/article_resources.html',{'article':article,'resources':resources},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_mediacentre')
def admin_article_add_resource(request,section_slug,article_slug):

    article = get_object_or_404(MediaArticle, slug=article_slug)
    form = MediaArticleResourceForm()

    if request.POST:
        form = MediaArticleResourceForm(request.POST)
        if form.is_valid():
            try:
                resource = form.save(commit=False)
                resource.article = article
                resource.save()
                messages.success(request,'Resource Created')
                return HttpResponseRedirect(reverse('admin_mediacentre_article_resources',args=[article.section.slug,article.slug]))
            except:
                messages.error(request,'Could not create resource')

    return render_to_response('admin/mediacentre/add_article_resource.html',{'article':article,'form':form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_mediacentre')
def admin_article_edit_resource(request,section_slug,article_slug,resourceid):

    article = get_object_or_404(MediaArticle, slug=article_slug)
    resource = get_object_or_404(MediaArticleResource,id=resourceid)

    form = MediaArticleResourceForm(instance=resource)

    if request.POST:
        form = MediaArticleResourceForm(request.POST,instance=resource)
        if form.is_valid():
            try:
                form.save()
                messages.success(request,'Resource Updated')
                return HttpResponseRedirect(reverse('admin_mediacentre_article_resources',args=[article.section.slug,article.slug]))
            except:
                messages.error(request,'Could not update resource')

    return render_to_response('admin/mediacentre/edit_article_resource.html',{'article':article,'resource':resource,'form':form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_mediacentre')
def admin_article_delete_resource(request,section_slug,article_slug,resourceid):

    article = get_object_or_404(MediaArticle, slug=article_slug)
    resource = get_object_or_404(MediaArticleResource,id=resourceid)

    try:
        resource.delete()
        messages.success(request,'Resource Deleted')
    except:
        messages.error(request,'Could not delete resource')

    return HttpResponseRedirect(reverse('admin_mediacentre_article_resources',args=[article.section.slug,article.slug]))
