#Django core bits
from django.shortcuts import render_to_response, get_object_or_404
from django.contrib.auth.decorators import login_required, permission_required
from django.conf import settings
from django.http import HttpResponseRedirect, Http404, HttpResponse
from django.core.urlresolvers import reverse
from django.contrib import messages
from datetime import datetime, date, time
import csv

from modules.core.decorators import *
from modules.core.functions import *

#Forms
from modules.meeting_booking.forms import *
from modules.members.forms import *

#Models
from modules.meeting_booking.models import *
from modules.members.models import *
from modules.payments.models import *

from modules.meeting_booking.signals import *

@permission_required('admin_users.can_access_meeting_booking')
def meeting_reports(request,meeting_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    sessions = MeetingSession.objects.filter(meeting=meeting)
    booking_types = MeetingBookingType.objects.filter(meeting=meeting)
    social_events = MeetingSocialEvent.objects.filter(meeting=meeting).order_by('name')
       
    return render(request,'admin/meeting_booking/reports/reports.html',{'meeting':meeting,'sessions':sessions,'booking_types':booking_types,'social_events':social_events})
    
    
@permission_required('admin_users.can_access_meeting_booking')
def download_delegates(request,meeting_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    
    meeting_bookings = MeetingBooking.objects.filter(complete=True,status='approved',meeting=meeting).order_by('-time')
    
    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_delegates.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Name','Job Title','Place of Work'])
    
    for meeting_booking in meeting_bookings:
        
        writer.writerow([unicode(s).encode("utf-8") for s in ("%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.job_title,meeting_booking.institution)])

    return response
    
@permission_required('admin_users.can_access_meeting_booking')
def download_bookings(request,meeting_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    
    meeting_bookings = MeetingBooking.objects.filter(complete=True,status='approved',meeting=meeting).order_by('-time')
    
    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_bookings.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Title','Given Name','Surname','Email Address','Created','Institution','Job Title','Trust','Hospital','Badge Name', 'Badge Organisation', 'Address 1','Address 2','Town','County','Country','Postcode','Telephone', 'Requirements', 'Dietary Requirements', 'Other', 'Willing to Share?', 'Type','Status','Sessions','Days','Additional Items','Total Price', 'Discount Code', 'Discount', 'Paid','Invoice Organisation','Invoice Address 1','Invoice Address 2','Invoice Address 3','Invoice Town','Invoice Country','Invoice Postcode','Invoice PO Number','Invoice Email','Invoice Phone'])
    
    for meeting_booking in meeting_bookings:

        reqs = ''
        for dietary_requirement in meeting_booking.dietary_requirements.all():
            reqs = "%s%s," % (reqs, dietary_requirement)
        
        sessions = meeting_booking.get_session_bookings()
        days    = meeting_booking.get_meeting_day_bookings()
        social_events = meeting_booking.get_social_events()
        
        session_text = ''
        for session in sessions:
            if session.waiting_list:
                session_text = "%s%s (Waiting List)," % (session_text,session.session_name)
            else:
                session_text = "%s%s," % (session_text,session.session_name)
                
        day_text = ''
        for day in days:
            day_text = "%s %s %s," % (day_text,day.meeting_day_name,day.price_paid)
            
        events_text = ''
        for event in social_events:
            events_text = "%s %sx %s %s," % (events_text,event.quantity,event.social_event_name,event.price_paid)
        
        invoice = meeting_booking.get_invoice() 
        
        if meeting_booking.paid:
            paid = 'Yes'
        else:
            paid = 'No'

        if meeting_booking.sharing:
            sharing = 'Yes'
        else:
            sharing = 'No'
        
        if invoice:
            writer.writerow([unicode(s).encode("utf-8") for s in (meeting_booking.title,meeting_booking.given_name,meeting_booking.surname,meeting_booking.email_address,meeting_booking.time.strftime('%H:%M %d/%m/%Y'),meeting_booking.institution,meeting_booking.job_title,meeting_booking.trust,meeting_booking.hospital, meeting_booking.badge_name, meeting_booking.badge_organisation, meeting_booking.address_1,meeting_booking.address_2,meeting_booking.town,meeting_booking.county,meeting_booking.country,meeting_booking.postcode,meeting_booking.telephone, meeting_booking.requirements, reqs, meeting_booking.dietary_requirements_other, sharing, meeting_booking.type_name,meeting_booking.get_status_display(),session_text,day_text,events_text,meeting_booking.get_total(), meeting_booking.type_code, meeting_booking.discount, paid,invoice.organisation_name,invoice.address_1,invoice.address_2,invoice.address_3,invoice.town,invoice.country,invoice.postcode,invoice.po_number,invoice.email_address,invoice.telephone)])
        else:
            writer.writerow([unicode(s).encode("utf-8") for s in (meeting_booking.title,meeting_booking.given_name,meeting_booking.surname,meeting_booking.email_address,meeting_booking.time.strftime('%H:%M %d/%m/%Y'),meeting_booking.institution,meeting_booking.job_title,meeting_booking.trust,meeting_booking.hospital, meeting_booking.badge_name, meeting_booking.badge_organisation,meeting_booking.address_1,meeting_booking.address_2,meeting_booking.town,meeting_booking.county,meeting_booking.country,meeting_booking.postcode,meeting_booking.telephone, meeting_booking.requirements, reqs, meeting_booking.dietary_requirements_other, sharing, meeting_booking.type_name,meeting_booking.get_status_display(),session_text,day_text,events_text,meeting_booking.type_code, meeting_booking.discount, meeting_booking.get_total(),meeting_booking.type_code, meeting_booking.discount, paid)])

    return response
    
@permission_required('admin_users.can_access_meeting_booking')
def download_badges(request,meeting_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    
    meeting_bookings = MeetingBooking.objects.filter(complete=True,status='approved',meeting=meeting).order_by('-time')
    
    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_booking_badges.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Name','Hospital/Organisation','Town','Registrant Type'])

    for meeting_booking in meeting_bookings:
        
        if meeting_booking.badge_name:
            name = meeting_booking.badge_name
        else:
            name= "%s %s %s" % (meeting_booking.title,meeting_booking.given_name,meeting_booking.surname)

        if meeting_booking.badge_organisation:
            organisation = meeting_booking.badge_organisation
        else:
            organisation = meeting_booking.institution

        writer.writerow([unicode(s).encode("utf-8") for s in (name,organisation,meeting_booking.town,meeting_booking.get_registrant_type_display())])

    return response
    
@permission_required('admin_users.can_access_meeting_booking')
def download_dietary_requirements(request,meeting_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    meeting_bookings = MeetingBooking.objects.filter(complete=True,status='approved',meeting=meeting).order_by('-time')
    
    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_booking_dietary_requirements.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Membership Number','Name','Email','Dietary Requirements', 'Other'])

    for meeting_booking in meeting_bookings:
        
        reqs = ''
        for dietary_requirement in meeting_booking.dietary_requirements.all():
            reqs = "%s%s," % (reqs, dietary_requirement)
        
        writer.writerow([unicode(s).encode("utf-8") for s in (meeting_booking.get_membership_number(),"%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.email_address,reqs, meeting_booking.dietary_requirements, meeting_booking.dietary_requirements_other)])

    return response
    
@permission_required('admin_users.can_access_meeting_booking')
def still_owing(request,meeting_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    
    meeting_bookings = MeetingBooking.objects.filter(complete=True,paid=False,meeting=meeting).order_by('-time')
    
    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_still_owing.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Name','Membership Number','Amount Owing'])
    
    for meeting_booking in meeting_bookings:
        
        writer.writerow([unicode(s).encode("utf-8") for s in ("%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.get_membership_number(),meeting_booking.get_amount_outstanding())])

    return response
    
    
@permission_required('admin_users.can_access_meeting_booking')
def invoices(request,meeting_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    
    invoices = MeetingBookingInvoice.objects.filter(meeting_booking__complete=True,meeting_booking__meeting=meeting).order_by('-meeting_booking__time')
    
    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_invoices.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Name','Membership Number','Amount Owing','Paid','Invoice Organisation','Invoice Address 1','Invoice Address 2','Invoice Address 3','Invoice Town','Invoice Country','Invoice Postcode','Invoice PO Number','Invoice Email','Invoice Phone'])
    
    for invoice in invoices:
        
        meeting_booking = invoice.meeting_booking
        if meeting_booking.paid:
            paid ='Yes'
        else:
            paid = 'No'
        
        if invoice:
            writer.writerow([unicode(s).encode("utf-8") for s in ("%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.get_membership_number(),meeting_booking.get_amount_outstanding(),paid,invoice.organisation_name,invoice.address_1,invoice.address_2,invoice.address_3,invoice.town,invoice.country,invoice.postcode,invoice.po_number,invoice.email_address,invoice.telephone)])

    return response

@permission_required('admin_users.can_access_meeting_booking')
def download_session(request,meeting_id,session_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    session = get_object_or_404(MeetingSession,id=session_id)
    session_bookings = MeetingSessionBooking.objects.filter(session=session,meeting_booking__status='approved')
    
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_session_delegates.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Membership Number','Name','Email'])
    
    for delegate in session_bookings:
        
        meeting_booking = delegate.meeting_booking
        
        writer.writerow([unicode(s).encode("utf-8") for s in (meeting_booking.get_membership_number(),"%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.email_address)])

    return response
    
@permission_required('admin_users.can_access_meeting_booking')
def download_booking_type(request,meeting_id,type_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    type = get_object_or_404(MeetingBookingType,id=type_id)
    meeting_bookings = MeetingBooking.objects.filter(type=type,status='approved')
   
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_type_delegates.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Membership Number','Name','Email'])
    
    for meeting_booking in meeting_bookings:
        
        writer.writerow([unicode(s).encode("utf-8") for s in (meeting_booking.get_membership_number(),"%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.email_address)])

    return response
    
@permission_required('admin_users.can_access_meeting_booking')
def download_social_event(request,meeting_id,social_event_id):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    social_event = get_object_or_404(MeetingSocialEvent,id=social_event_id)
    social_event_bookings = MeetingSocialEventBooking.objects.filter(social_event=social_event,meeting_booking__status='approved')
    
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_social_event_delegates.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Membership Number','Name','Email'])
    
    for social_event_booking in social_event_bookings:
        
        meeting_booking = social_event_booking.meeting_booking
        
        writer.writerow([unicode(s).encode("utf-8") for s in (meeting_booking.get_membership_number(),"%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.email_address)])

    return response
    
    
@permission_required('admin_users.can_access_meeting_booking')
def download_member_type(request,meeting_id,type_slug):
    
    meeting = get_object_or_404(Meeting,id=meeting_id)
    meeting_bookings = MeetingBooking.objects.filter(registrant_type=type_slug,status='approved')
   
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_member_type.csv"

    writer = csv.writer(response)
    
    writer.writerow(['Membership Number','Name','Email'])
    
    for meeting_booking in meeting_bookings:
        
        writer.writerow([unicode(s).encode("utf-8") for s in (meeting_booking.get_membership_number(),"%s %s" % (meeting_booking.given_name,meeting_booking.surname),meeting_booking.email_address)])

    return response