from django import forms
from models import *
from modules.pages.models import Page
from modules.meeting_booking.models import MeetingBooking
from modules.members.models import Member

from django.db.models import Q

class BioForm(forms.ModelForm):

    class Meta:
        model = Bio
        fields = ('name', 'email_address', 'image', 'description')

class InviteBiosForm(forms.Form):

    emails = forms.CharField(widget=forms.Textarea)
    message = forms.CharField(widget=forms.Textarea)

class MeetingInformationForm(forms.ModelForm):

    class Meta:
        model = MeetingInformation
        fields = ('name','long_name','enabled','content', 'meeting_booking')


class SplashPageForm(forms.ModelForm):

    class Meta:
        model = MeetingInformation
        fields = ('splash_enabled', 'splash_title', 'splash_image','splash_content')

class MeetingLinkForm(forms.ModelForm):

    class Meta:
        model = MeetingLink
        exclude = ('meeting',)

class MeetingPageForm(forms.ModelForm):
    class Meta:
        model = MeetingPage
        exclude = ('meeting','url','full_url')

class ImportPageForm(forms.Form):
    title = forms.CharField(max_length=200)
    parent = forms.ModelChoiceField(queryset=MeetingPage.objects.all(),required=False)
    page = forms.ModelChoiceField(queryset=Page.objects.all(),empty_label=None)

class MeetingPageEditForm(forms.ModelForm):

    class Meta:
        model = MeetingPage
        exclude = ('meeting','full_url',)

class MeetingBioPageForm(forms.ModelForm):
    class Meta:
        model = MeetingBioPage
        exclude = ('meeting','url')

class MeetingBioPageEditForm(forms.ModelForm):

    class Meta:
        model = MeetingBioPage
        exclude = ('meeting',)

class MeetingPageResourceForm(forms.ModelForm):
    class Meta:
        model = MeetingPageResource
        exclude = ('page',)

class MeetingFloorPlanForm(forms.ModelForm):
    class Meta:
        model = MeetingFloorPlan
        exclude = ('meeting',)

class MeetingBannerForm(forms.ModelForm):
    class Meta:
        model = MeetingBanner
        exclude = ('meeting',)

class ProgrammeSessionForm(forms.ModelForm):

    class Meta:
        model = ProgrammeSession
        exclude = ('meeting',)

    def __init__(self, *args, **kwargs):
        super(ProgrammeSessionForm, self).__init__(*args, **kwargs)
        #self.fields['sessions'].label_from_instance = lambda obj: "%s - %s" % (obj.meeting,obj.name)

class ProgrammeSessionMultipleForm(forms.ModelForm):

    class Meta:
        model = ProgrammeSession
        fields = ('name','day','start_time','end_time', 'time_of_day', 'programme_name','programme_description','programme_presenters','session_type','primary_subject','location','chair','chair_2')
        widgets = {
            'programme_description': forms.Textarea(attrs={'cols': 20, 'rows': 2}),
            'programme_presenters': forms.Textarea(attrs={'cols': 20, 'rows': 2}),
        }

class ProgrammeSessionFormSet(forms.BaseModelFormSet):
    def __init__(self, days, locations, contacts, *args, **kwargs):
        super(ProgrammeSessionFormSet, self).__init__(*args, **kwargs)

        for form in self.forms:
            form.fields['day'].queryset = days
            form.fields['location'].queryset = locations
            form.fields['chair'].queryset = contacts
            form.fields['chair_2'].queryset = contacts

class ProgrammeSessionEmptyForm(ProgrammeSessionMultipleForm):

    def __init__(self, days, locations, contacts, *args, **kwargs):
        super(ProgrammeSessionEmptyForm, self).__init__(*args, **kwargs)

        self.fields['day'].queryset = days
        self.fields['location'].queryset = locations
        self.fields['chair'].queryset = contacts
        self.fields['chair_2'].queryset = contacts

class SessionPresentationForm(forms.ModelForm):

    class Meta:
        model = SessionPresentation
        fields = ('title','order','speaker','speaker_2', 'speaker_3', 'speaker_4', 'speaker_5')

class SessionPresentationMultipleForm(forms.ModelForm):

    class Meta:
        model = SessionPresentation
        fields = ('title','meeting_session','order','speaker','speaker_2', 'speaker_3', 'speaker_4', 'speaker_5')

class SessionPresentationFormSet(forms.BaseModelFormSet):
    def __init__(self, sessions,abstracts,contacts, *args, **kwargs):
        super(SessionPresentationFormSet, self).__init__(*args, **kwargs)

        for form in self.forms:
            form.fields['meeting_session'].queryset = sessions
            #form.fields['abstract_submission'].queryset = abstracts
            form.fields['speaker'].queryset = contacts
            form.fields['speaker_2'].queryset = contacts
            form.fields['speaker_3'].queryset = contacts
            form.fields['speaker_4'].queryset = contacts
            form.fields['speaker_5'].queryset = contacts

class SessionPresentationEmptyForm(SessionPresentationMultipleForm):

    def __init__(self, sessions,abstracts,contacts, *args, **kwargs):
        super(SessionPresentationEmptyForm, self).__init__(*args, **kwargs)

        self.fields['meeting_session'].queryset = sessions
        #self.fields['abstract_submission'].queryset = abstracts
        self.fields['speaker'].queryset = contacts
        self.fields['speaker_2'].queryset = contacts
        self.fields['speaker_3'].queryset = contacts
        self.fields['speaker_4'].queryset = contacts
        self.fields['speaker_5'].queryset = contacts

class SessionDayForm(forms.ModelForm):

    class Meta:
        model = SessionDay
        fields = ('date',)

class SessionLocationForm(forms.ModelForm):

    class Meta:
        model = SessionLocation
        fields = ('name',)

class SessionTypeForm(forms.ModelForm):

    class Meta:
        model = SessionType
        fields = ('name',)

class SessionSubjectForm(forms.ModelForm):

    class Meta:
        model = SessionSubject
        fields = ('name',)

class MeetingContactForm(forms.ModelForm):

    def __init__(self, *args, **kwargs):
        super(MeetingContactForm, self).__init__(*args, **kwargs)
        for field in self.fields:
            self.fields[field].required = False

    class Meta:
        model = MeetingContact
        exclude = ('meeting',)

class MeetingMemberContactForm(forms.Form):

    members = forms.ModelMultipleChoiceField(queryset=Member.objects.filter(member_status='current', user_type='member'))

class MeetingEmailerForm(forms.ModelForm):

    def __init__(self, *args, **kwargs):
        super(MeetingEmailerForm, self).__init__(*args, **kwargs)
        self.fields['member'].queryset = Member.objects.filter(Q(member_status='current', user_type='member') | Q(user_type='non-member'))

    class Meta:
        model = MeetingEmailer
        exclude = ('meeting',)

class MeetingPosterForm(forms.ModelForm):

    class Meta:
        model = MeetingPoster
        fields = ('title','poster_type','poster_file','abstract_submission','contact')

class MeetingExhibitorForm(forms.ModelForm):

    class Meta:
        model = MeetingExhibitor
        fields = ('name','image','description','booth', 'floor_plan')

class MeetingVoteForm(forms.ModelForm):

    class Meta:
        model = MeetingVote
        fields = ('name','voting_type','start_time','end_time', 'show_results')

class MeetingVoteReminderForm(forms.ModelForm):

    class Meta:
        model = MeetingVoteReminder
        fields = ('subject', 'content',)

class VoteQuestionMultipleForm(forms.ModelForm):

    class Meta:
        model = MeetingVoteQuestion
        fields = ('name','options','order')

class MeetingEvaluatorForm(forms.ModelForm):

    def __init__(self, *args, **kwargs):
        from modules.members.models import Member
        super(MeetingEvaluatorForm, self).__init__(*args, **kwargs)
        self.fields['evaluators'].required = True
        meeting_booking = self.instance.meeting_booking
        if meeting_booking:
            booking_member_ids = MeetingBooking.objects.filter(meeting=meeting_booking, complete=True).values_list('member_id', flat=True)
            self.fields['evaluators'].queryset = Member.objects.filter(id__in=booking_member_ids)
        else:
            self.fields['evaluators'].queryset = Member.objects.filter(approved=True, complete=True)

    class Meta:
        model = MeetingInformation
        fields = ('evaluators',)
