from django.shortcuts import render,redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, permission_required
from django.contrib import messages
from django.forms.formsets import formset_factory
from django.forms import modelformset_factory
from django.db.models import Q
from django.conf import settings
from django.core.urlresolvers import reverse
import csv

from django.template import Context, Template

from django.template.loader import render_to_string
from django.http import HttpResponse

from models import *
from adminforms import *
from emails import send_bio_request_email, vote_reminder

from modules.abstracts.models import AbstractSubmission
from modules.meeting_booking.models import MeetingBooking

from modules.core.functions import slugify_unique, random_string_unique, is_valid_email

@permission_required('admin_users.can_access_meeting_information')
def bios(request):

    completed_bios = Bio.objects.filter(details_entered=True)
    pending_bios = Bio.objects.filter(details_entered=False)

    return render(request,'admin/bios/bios.html',{'completed_bios':completed_bios, 'pending_bios':pending_bios})

@permission_required('admin_users.can_access_meeting_information')
def invite_bios(request):

    invite_form = InviteBiosForm()

    if request.POST:
        invite_form = InviteBiosForm(request.POST, request.FILES)
        if invite_form.is_valid():

            post_emails = invite_form.cleaned_data['emails']
            records = post_emails.split('\r\n')
            post_message = invite_form.cleaned_data['message']

            errors = []

            line = 1
            for record in records:

                if not is_valid_email(record):
                    messages.error(request, 'Line %s is not a valid Email Address' % (line))
                    errors = True

                line+= 1

            if not errors:
                for record in records:
                    unique_key = random_string_unique(20, Bio, 'unique_key')
                    profile_url = "%s%s" % (settings.URL, reverse('meeting_information_submit_bio', args=[unique_key]))

                    t_message = Template(post_message)
                    message = t_message.render(Context({
                        'profile_url': profile_url
                    }))

                    bio = Bio(email_address=record, unique_key=unique_key, message=message)
                    """try:
                        bio.name = record_details[0]
                    except:
                        pass"""
                    bio.save()

                    send_bio_request_email(bio)

                messages.success(request, 'Bios have been created')
                return redirect('admin_meeting_information_bios')

    return render(request, 'admin/bios/invite-bios.html', {'invite_form':invite_form})

@permission_required('admin_users.can_access_meeting_information')
def add_bio(request):

    bio_form = BioForm()

    if request.POST:
        bio_form = BioForm(request.POST, request.FILES)

        if bio_form.is_valid():
            bio = bio_form.save(commit=False)
            bio.details_entered = True
            bio.save()

            messages.success(request, 'Bio has been created')
            return redirect('admin_meeting_information_bios')

    return render(request,'admin/bios/add-bio.html',{'bio_form':bio_form})

@permission_required('admin_users.can_access_meeting_information')
def edit_bio(request, bio_id):

    bio = get_object_or_404(Bio, id=bio_id)
    bio_form = BioForm(instance=bio)

    if request.POST:
        bio_form = BioForm(request.POST, request.FILES, instance=bio)

        if bio_form.is_valid():
            bio_form.save()

            bio.details_entered = True
            bio.save()

            messages.success(request, 'Bio has been created')
            return redirect('admin_meeting_information_bios')

    return render(request,'admin/bios/edit-bio.html',{'bio':bio, 'bio_form':bio_form})

@permission_required('admin_users.can_access_meeting_information')
def delete_bio(request, bio_id):

    bio = get_object_or_404(Bio, id=bio_id)
    bio.delete()

    messages.success(request, 'Bio has been created')
    return redirect('admin_meeting_information_bios')

@permission_required('admin_users.can_access_meeting_information')
def resend_bio(request, bio_id):

    bio = get_object_or_404(Bio, id=bio_id)
    send_bio_request_email(bio)

    messages.success(request, 'Bio request has been resent')
    return redirect('admin_meeting_information_bios')

@permission_required('admin_users.can_access_meeting_information')
def meetings(request):

    enabled_meetings = MeetingInformation.objects.filter(enabled=True)
    disabled_meetings = MeetingInformation.objects.filter(enabled=False)

    return render(request,'admin/meeting_information/meetings.html',{'enabled_meetings':enabled_meetings,'disabled_meetings':disabled_meetings})

@permission_required('admin_users.can_access_meeting_information')
def add_meeting(request):

    meeting_form = MeetingInformationForm()

    if request.POST:
        meeting_form = MeetingInformationForm(request.POST,request.FILES)

        if meeting_form.is_valid():
            meeting = meeting_form.save(commit=False)
            meeting.slug = slugify_unique(meeting.name,MeetingInformation,'slug')
            meeting.save()
            meeting_form.save_m2m()

            messages.success(request,'Meeting Information has been created')
            return redirect('admin_meeting_information')

    return render(request,'admin/meeting_information/add-meeting.html',{'meeting_form':meeting_form})

@permission_required('admin_users.can_access_meeting_information')
def edit_meeting(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    meeting_form = MeetingInformationForm(instance=meeting)

    if request.POST:
        meeting_form = MeetingInformationForm(request.POST,request.FILES,instance=meeting)

        if meeting_form.is_valid():
            meeting_form.save()
            messages.success(request,'Meeting Information has been updated')
            return redirect('admin_meeting_information')

    return render(request,'admin/meeting_information/edit-meeting.html',{'meeting':meeting,'meeting_form':meeting_form})

@permission_required('admin_users.can_access_meeting_information')
def splash_page(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    splash_page_form = SplashPageForm(instance=meeting)

    if request.POST:
        splash_page_form = SplashPageForm(request.POST, request.FILES, instance=meeting)

        if splash_page_form.is_valid():
            splash_page_form.save()
            messages.success(request,'Splash Page has been updated')
            return redirect('admin_meeting_information')

    return render(request,'admin/meeting_information/splash-page.html',{'meeting':meeting,'splash_page_form':splash_page_form})

@permission_required('admin_users.can_access_meeting_information')
def duplicate_meeting(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    meeting_form = MeetingInformationForm(instance=meeting)

    if request.POST:
        meeting_form = MeetingInformationForm(request.POST, request.FILES)

        if meeting_form.is_valid():
            new_meeting = meeting_form.save(commit=False)
            new_meeting.save()
            meeting_form.save_m2m()

            for day in meeting.get_days():
                programme_day = SessionDay(meeting=new_meeting,date=day.date)
                programme_day.save()

            for location in meeting.get_locations():
                programme_location = SessionLocation(meeting=new_meeting, name=location.name)
                programme_location.save()

            for contact in meeting.get_contacts():
                contact.pk = None
                contact.meeting = new_meeting
                contact.save()

            sessions = meeting.get_sessions()
            for session in sessions:
                new_day = SessionDay.objects.get(meeting=new_meeting, date=session.day.date)
                if session.location:
                    new_location = SessionLocation.objects.get(meeting=new_meeting, name=session.location.name)
                else:
                    new_location = None
                if session.chair:
                    new_chairs = MeetingContact.objects.filter(meeting=new_meeting, first_name=session.chair.first_name, last_name=session.chair.last_name)                
                    if new_chairs:
                        new_chair = new_chairs[0]
                    else:
                        new_chair = None
                else:
                    new_chair = None
                if session.chair_2:
                    new_chairs_2 = MeetingContact.objects.filter(meeting=new_meeting, first_name=session.chair_2.first_name, last_name=session.chair_2.last_name)                
                    if new_chairs_2:
                        new_chair_2 = new_chairs_2[0]
                    else:
                        new_chair = None
                else:
                    new_chair_2 = None
                
                session.pk = None
                session.meeting = new_meeting
                session.day = new_day
                session.location = new_location
                session.chair = new_chair
                session.chair_2 = new_chair_2
                session.save()

            for page in meeting.get_all_pages():
                page.pk = None
                page.meeting = new_meeting
                page.save()

            for link in meeting.get_links():
                link.pk = None
                link.meeting = new_meeting
                link.save()
            
            for bio_page in meeting.get_bio_pages():
                bio_page.pk = None
                bio_page.meeting = new_meeting
                bio_page.save()

            for exhibitor in meeting.get_exhibitors():
                exhibitor.pk = None
                exhibitor.meeting = new_meeting
                exhibitor.save()
            
            messages.success(request, 'Meeting has been added')
            return redirect('admin_meeting_information')

    return render(request,'admin/meeting_information/duplicate-meeting.html',{'meeting':meeting,'meeting_form':meeting_form})

@permission_required('admin_users.can_access_meeting_information')
def delete_meeting(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    meeting.delete()

    messages.success(request,'Meeting Information has been deleted')
    return redirect('admin_meeting_information')

@permission_required('admin_users.can_access_meeting_information')
def enable_meeting(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    meeting.enabled=True
    meeting.save()

    messages.success(request,'Meeting Information has been enabled')
    return redirect('admin_meeting_information')

@permission_required('admin_users.can_access_meeting_information')
def disable_meeting(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    meeting.enabled=False
    meeting.save()

    messages.success(request,'Meeting Information has been disabled')
    return redirect('admin_meeting_information')

@permission_required('admin_users.can_access_meeting_information')
def pdf(request,meeting_id):

    import pdfcrowd

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    days = SessionDay.objects.filter(meeting=meeting)

    #return render(request,'admin/meeting_information/pdf.html',{'meeting':meeting,'meeting_days':days})

    html = render_to_string('admin/meeting_information/pdf.html',{'meeting':meeting,'meeting_days':days})

    client = pdfcrowd.Client("calmdigital", "dc33e7c4525620565185d0a00d90b8f0")
    client.setPageMargins('20','20','20','20')
    pdf = client.convertHtml(html)

    meeting.pdf = pdf
    meeting.save()

    messages.success(request,'PDF programme has been generated')

    return redirect('admin_meeting_information_programme',meeting_id=meeting_id)

@permission_required('admin_user.can_access_meeting_information')
def download_pdf(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    if not meeting.pdf:
        messages.error(request,'Sorry, the PDF has not been generated yet.')
        return redirect('admin_meeting_information_programme',meeting_id=meeting_id)
    else:
        response = HttpResponse(content_type="application/pdf")
        response["Cache-Control"] = "no-cache"
        response["Accept-Ranges"] = "none"
        response["Content-Disposition"] = "attachment; filename=meeting_programme.pdf"
        response.write(meeting.pdf)

        return response

@permission_required('admin_user.can_access_meeting_information')
def download_csv(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    fields = ['Day','Start Time','End Time','Name', 'Programme Name', 'Programme Description', 'Programme Presenters', 'Primary Subject', 'Secondary Subject', 'Session Type', 'Location', 'Chair', 'Chair 2']
    
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=meeting_programme.csv"

    writer = csv.writer(response)
    writer.writerow(fields)   

    for session in meeting.get_sessions():

        session_array = [
            session.day.d_format(),
            session.start_time,
            session.end_time,
            session.name,
            session.programme_name,
            session.programme_description,
            session.programme_presenters,
            session.primary_subject,
            session.secondary_subject,
            session.session_type,
            session.location,
            session.chair,
            session.chair_2
        ]

        writer.writerow([unicode(s).encode("utf-8") for s in session_array])

    return response

###########################LINKS###########################

@permission_required('admin_users.can_access_meeting_information')
def links(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    links = MeetingLink.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/links/links.html',{'meeting':meeting,'links': links})

@permission_required('admin_users.can_access_meeting_information')
def add_link(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingLinkForm()

    if request.POST:
        form = MeetingLinkForm(request.POST,request.FILES)
        if form.is_valid():
            link = form.save(commit=False)
            link.meeting = meeting
            link.save()

            messages.success(request,'Link Created')
            return redirect('admin_meeting_information_links',meeting_id=meeting.id)
    return render(request,'admin/meeting_information/links/add_link.html', {'meeting':meeting,'form': form})

@permission_required('admin_users.can_access_meeting_information')
def edit_link(request, meeting_id, link_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    link = get_object_or_404(MeetingLink,id=link_id)
    form = MeetingLinkForm(instance=link)
    if request.POST:
        form = MeetingLinkForm(request.POST,request.FILES,instance=link)

        if form.is_valid():
            form.save()
            messages.success(request,'Link Updated')

            return redirect('admin_meeting_information_links',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/links/edit_link.html', {'meeting':meeting,'form': form, 'link': link})

@permission_required('admin_users.can_access_meeting_information')
def delete_link(request,meeting_id,link_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    link = get_object_or_404(MeetingLink,id=link_id)
    link.delete()
    messages.success(request,'Link Deleted')

    return redirect('admin_meeting_information_links',meeting_id=meeting.id)


###########################PAGES###########################

@permission_required('admin_users.can_access_meeting_information')
def pages(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    pages = MeetingPage.objects.filter(parent__isnull=True,meeting=meeting).order_by('title')

    return render(request,'admin/meeting_information/pages/pages.html',{'meeting':meeting,'pages': pages})

@permission_required('admin_users.can_access_meeting_information')
def add_page(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingPageForm()
    form.fields['parent'].queryset = MeetingPage.objects.filter(meeting=meeting)

    if request.POST:
        form = MeetingPageForm(request.POST,request.FILES)
        form.fields['parent'].queryset = MeetingPage.objects.filter(meeting=meeting)
        if form.is_valid():
            page = form.save(commit=False)
            page.url = slugify_unique(page.title, MeetingPage,'url')
            page.meeting = meeting
            page.save()

            messages.success(request,'Page Created')
            return redirect('admin_meeting_information_pages',meeting_id=meeting.id)
    return render(request,'admin/meeting_information/pages/add_page.html', {'meeting':meeting,'form': form})

@permission_required('admin_users.can_access_meeting_information')
def import_page(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = ImportPageForm()

    if request.POST:
        form = ImportPageForm(request.POST)
        if form.is_valid():
            page = form.cleaned_data['page']
            title = form.cleaned_data['title']

            m_page = MeetingPage(
                meeting = meeting,
                meta_title = title,
                title = title,
                url = slugify_unique(title, MeetingPage,'url'),
                parent = form.cleaned_data['parent'],
                introduction = page.introduction,
                introduction_image = page.introduction_image,
                content = page.content,
            )
            m_page.save()

            messages.success(request,'Page Created')
            return redirect('admin_meeting_information_pages',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/pages/import_page.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_page(request, meeting_id, page_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingPage,id=page_id)
    form = MeetingPageEditForm(instance=page)
    form.fields['parent'].queryset  = MeetingPage.objects.filter(meeting=meeting).exclude(id=page.id)
    if request.POST:
        form = MeetingPageEditForm(request.POST,request.FILES,instance=page)
        form.fields['parent'].queryset  = MeetingPage.objects.filter(meeting=meeting).exclude(id=page.id)

        if form.is_valid():
            form.save()
            messages.success(request,'Page Updated')

            return redirect('admin_meeting_information_pages',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/pages/edit_page.html', {'meeting':meeting,'form': form, 'page': page})

@permission_required('admin_users.can_access_meeting_information')
def delete_page(request,meeting_id,page_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingPage, id=page_id)
    page.delete()
    messages.success(request,'Page Deleted')

    return redirect('admin_meeting_information_pages',meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def page_resources(request,meeting_id,page_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingPage, id=page_id)
    resources = MeetingPageResource.objects.filter(page=page).order_by('order')

    return render(request,'admin/meeting_information/pages/page_resources.html',{'meeting':meeting,'page':page,'resources':resources})

@permission_required('admin_users.can_access_meeting_information')
def add_resource_page(request,meeting_id,page_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingPage, id=page_id)

    form = MeetingPageResourceForm()

    if request.POST:
        form = MeetingPageResourceForm(request.POST)
        if form.is_valid():
            resource = form.save(commit=False)
            resource.page = page
            resource.save()
            messages.success(request,'Resource Created')
            return redirect('admin_meeting_information_resources',meeting_id=meeting.id,page_id=page.id)

    return render(request,'admin/meeting_information/pages/add_page_resource.html',{'meeting':meeting,'page':page,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_resource_page(request,meeting_id,page_id,resource_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingPage,id=page_id)
    resource = get_object_or_404(MeetingPageResource,id=resource_id)

    form = MeetingPageResourceForm(instance=resource)

    if request.POST:
        form = MeetingPageResourceForm(request.POST,instance=resource)
        if form.is_valid():
            form.save()
            messages.success(request,'Resource Updated')

            return redirect('admin_meeting_information_resources',meeting_id=meeting.id,page_id=page.id)

    return render(request,'admin/meeting_information/pages/edit_page_resource.html',{'meeting':meeting,'page':page,'resource':resource,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_resource_page(request,meeting_id,page_id,resource_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingPage,id=page_id)
    resource = get_object_or_404(MeetingPageResource,id=resource_id)

    resource.delete()
    messages.success(request,'Resource Deleted')

    return redirect('admin_meeting_information_resources',meeting_id=meeting.id,page_id=page.id)

###########################Bio Pages#############################

@permission_required('admin_users.can_access_meeting_information')
def bio_pages(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    pages = MeetingBioPage.objects.filter(meeting=meeting).order_by('title')

    return render(request,'admin/meeting_information/bio-pages/pages.html',{'meeting':meeting,'pages': pages})

@permission_required('admin_users.can_access_meeting_information')
def add_bio_page(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingBioPageForm()
    form.fields['bios'].queryset = Bio.objects.filter(details_entered=True)

    if request.POST:
        form = MeetingBioPageForm(request.POST,request.FILES)
        form.fields['bios'].queryset = Bio.objects.filter(details_entered=True)

        if form.is_valid():
            page = form.save(commit=False)
            page.url = slugify_unique(page.title, MeetingBioPage,'url')
            page.meeting = meeting
            page.save()

            form.save_m2m()

            messages.success(request,'Bio Page Created')
            return redirect('admin_meeting_information_bio_pages', meeting_id=meeting.id)

    return render(request,'admin/meeting_information/bio-pages/add_page.html', {'meeting':meeting,'form': form})

@permission_required('admin_users.can_access_meeting_information')
def edit_bio_page(request, meeting_id, page_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingBioPage,id=page_id)
    form = MeetingBioPageEditForm(instance=page)

    if request.POST:
        form = MeetingBioPageEditForm(request.POST,request.FILES,instance=page)

        if form.is_valid():
            form.save()
            messages.success(request,'Page Updated')

            return redirect('admin_meeting_information_bio_pages', meeting_id=meeting.id)

    return render(request,'admin/meeting_information/bio-pages/edit_page.html', {'meeting':meeting,'form': form, 'page': page})

@permission_required('admin_users.can_access_meeting_information')
def delete_bio_page(request,meeting_id,page_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    page = get_object_or_404(MeetingBioPage, id=page_id)
    page.delete()
    messages.success(request,'Page Deleted')

    return redirect('admin_meeting_information_bio_pages', meeting_id=meeting.id)

##########################Floor Plans############################
@permission_required('admin_users.can_access_meeting_information')
def floor_plans(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    plans = MeetingFloorPlan.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/floor_plans/floor_plans.html',{'meeting':meeting,'floor_plans': plans})

@permission_required('admin_users.can_access_meeting_information')
def add_floor_plan(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingFloorPlanForm()

    if request.POST:
        form = MeetingFloorPlanForm(request.POST,request.FILES)
        if form.is_valid():
            plan = form.save(commit=False)
            plan.meeting = meeting
            plan.save()

            messages.success(request,'Plan Created')
            return redirect('admin_meeting_information_floor_plans',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/floor_plans/add_floor_plan.html', {'meeting':meeting,'form': form})

@permission_required('admin_users.can_access_meeting_information')
def edit_floor_plan(request, meeting_id, plan_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    plan = get_object_or_404(MeetingFloorPlan,id=plan_id)
    form = MeetingFloorPlanForm(instance=plan)
    if request.POST:
        form = MeetingFloorPlanForm(request.POST,request.FILES,instance=plan)

        if form.is_valid():
            form.save()
            messages.success(request,'Plan Updated')

            return redirect('admin_meeting_information_floor_plans',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/floor_plans/edit_floor_plan.html', {'meeting':meeting,'form': form, 'plan': plan})

@permission_required('admin_users.can_access_meeting_information')
def delete_floor_plan(request,meeting_id,plan_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    plan = get_object_or_404(MeetingFloorPlan,id=plan_id)
    plan.delete()
    messages.success(request,'Plan Deleted')

    return redirect('admin_meeting_information_floor_plans',meeting_id=meeting.id)


##########################Banners############################
@permission_required('admin_users.can_access_meeting_information')
def banners(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    banners = MeetingBanner.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/banners/banners.html',{'meeting':meeting,'banners': banners})

@permission_required('admin_users.can_access_meeting_information')
def add_banner(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingBannerForm()

    if request.POST:
        form = MeetingBannerForm(request.POST,request.FILES)
        if form.is_valid():
            banner = form.save(commit=False)
            banner.meeting = meeting
            banner.save()

            messages.success(request,'Banner Created')
            return redirect('admin_meeting_information_banners',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/banners/add_banner.html', {'meeting':meeting,'form': form})

@permission_required('admin_users.can_access_meeting_information')
def edit_banner(request, meeting_id, banner_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    banner = get_object_or_404(MeetingBanner,id=banner_id)
    form = MeetingBannerForm(instance=banner)
    if request.POST:
        form = MeetingBannerForm(request.POST,request.FILES,instance=banner)

        if form.is_valid():
            form.save()
            messages.success(request,'Banner Updated')

            return redirect('admin_meeting_information_banners', meeting_id=meeting.id)

    return render(request,'admin/meeting_information/banners/edit_banner.html', {'meeting':meeting,'form': form, 'banner': banner})

@permission_required('admin_users.can_access_meeting_information')
def delete_banner(request,meeting_id,banner_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    banner = get_object_or_404(MeetingBanner,id=banner_id)
    banner.delete()
    messages.success(request,'Banner Deleted')

    return redirect('admin_meeting_information_banners', meeting_id=meeting.id)


#############################programme###########################

@permission_required('admin_users.can_access_meeting_information')
def programme(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    days = SessionDay.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/programme/programme.html',{'meeting':meeting,'days':days})


@permission_required('admin_users.can_access_meeting_information')
def add_programme_session(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = ProgrammeSessionForm()
    form.fields['day'].queryset = SessionDay.objects.filter(meeting=meeting)
    form.fields['location'].queryset = SessionLocation.objects.filter(meeting=meeting)
    form.fields['chair'].queryset = form.fields['chair_2'].queryset = MeetingContact.objects.filter(meeting=meeting)

    if request.POST:
        form = ProgrammeSessionForm(request.POST)
        form.fields['day'].queryset = SessionDay.objects.filter(meeting=meeting)
        form.fields['location'].queryset = SessionLocation.objects.filter(meeting=meeting)
        form.fields['chair'].queryset = form.fields['chair_2'].queryset = MeetingContact.objects.filter(meeting=meeting)

        if form.is_valid():
            try:
                programme_session = form.save(commit=False)
                programme_session.meeting = meeting
                programme_session.save()

                messages.success(request,'Session Created')

                return redirect('admin_meeting_information_programme',meeting_id=meeting.id)

            except:
                messages.error(request,'Could not create Session')

    return render(request,'admin/meeting_information/programme/add-session.html',{'meeting':meeting,'form':form})


@permission_required('admin_users.can_access_meeting_information')
def manage_sessions(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    days = SessionDay.objects.filter(meeting=meeting)
    locations = SessionLocation.objects.filter(meeting=meeting)
    contacts = MeetingContact.objects.filter(meeting=meeting)

    PSFS = modelformset_factory(ProgrammeSession, form=ProgrammeSessionMultipleForm, can_delete=True, extra=1, formset=ProgrammeSessionFormSet)
    session_formset = PSFS(days,locations,contacts, queryset=ProgrammeSession.objects.filter(meeting=meeting))

    psef = ProgrammeSessionEmptyForm(days, locations, contacts,  prefix='__prefix__')

    if request.POST:
        session_formset = PSFS(days, locations, contacts, request.POST, queryset=ProgrammeSession.objects.filter(meeting=meeting))
        if session_formset.is_valid():
            for session_form in session_formset:

                if session_form.cleaned_data:
                    session = session_form.save(commit=False)
                    session.meeting = meeting
                    session.save()

            for deleted in session_formset.deleted_forms:
                deleted.instance.delete()

            messages.success(request,'Forms have been saved')
            return redirect('admin_meeting_information_manage_sessions',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/programme/add-multiple-sessions.html',{'meeting':meeting,'session_formset':session_formset, 'psef':psef})


@permission_required('admin_users.can_access_meeting_information')
def edit_programme_session(request,meeting_id,session_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    programme_session = get_object_or_404(ProgrammeSession,id=session_id)

    form = ProgrammeSessionForm(instance=programme_session)
    form.fields['day'].queryset = SessionDay.objects.filter(meeting=meeting)
    form.fields['location'].queryset = SessionLocation.objects.filter(meeting=meeting)
    form.fields['chair'].queryset = form.fields['chair_2'].queryset = MeetingContact.objects.filter(meeting=meeting)

    if request.POST:
        form = ProgrammeSessionForm(request.POST,instance=programme_session)
        form.fields['day'].queryset = SessionDay.objects.filter(meeting=meeting)
        form.fields['location'].queryset = SessionLocation.objects.filter(meeting=meeting)
        form.fields['chair'].queryset = form.fields['chair_2'].queryset = MeetingContact.objects.filter(meeting=meeting)

        if form.is_valid():
            try:
                form.save()
                messages.success(request,'Session Updated')

                return redirect('admin_meeting_information_programme',meeting_id=meeting.id)

            except:
                messages.error(request,'Could not update Session')

    return render(request,'admin/meeting_information/programme/edit-session.html',{'meeting':meeting,'programme_session':programme_session,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_programme_session(request,meeting_id,session_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)

    try:
        session.delete()
        messages.success(request,'Session Deleted')

    except:
        messages.errror(request,'Could not delete Session')

    return redirect('admin_meeting_information_programme',meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def add_presentations(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    sessions = ProgrammeSession.objects.filter(meeting=meeting)
    contacts = MeetingContact.objects.filter(meeting=meeting)

    if meeting.abstract_request:
        abstracts = AbstractSubmission.objects.filter(abstract_request=meeting.abstract_request,status='approved-op')
    else:
        abstracts = AbstractSubmission.objects.none()

    SPFS = modelformset_factory(SessionPresentation, form=SessionPresentationMultipleForm, extra=1 , can_delete=True, formset=SessionPresentationFormSet)
    presentation_formset = SPFS(sessions,abstracts,contacts, queryset=SessionPresentation.objects.filter(meeting_session__meeting=meeting))

    spef = SessionPresentationEmptyForm(sessions, abstracts, contacts, prefix='__prefix__')

    if request.POST:
        presentation_formset = SPFS(sessions, abstracts, contacts, request.POST, queryset=SessionPresentation.objects.filter(meeting_session__meeting=meeting))

        if presentation_formset.is_valid():
            for presentation_form in presentation_formset:
                if presentation_form.cleaned_data:
                    presentation = presentation_form.save(commit=False)
                    presentation.save()

            for deleted in presentation_formset.deleted_forms:
                deleted.instance.delete()

            messages.success(request,'Forms have been saved')
            return redirect('admin_meeting_information_add_presentations', meeting_id=meeting.id)

    return render(request,'admin/meeting_information/programme/add-presentations.html',{'meeting':meeting,'presentation_formset':presentation_formset, 'spef':spef})

@permission_required('admin_users.can_access_meeting_information')
def session_presentations(request,meeting_id,session_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)
    presentations = SessionPresentation.objects.filter(meeting_session=session)

    return render(request,'admin/meeting_information/programme/presentations/presentations.html',{'meeting':meeting,'session':session,'presentations':presentations})

@permission_required('admin_users.can_access_meeting_information')
def add_session_presentation(request,meeting_id,session_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)

    contacts = MeetingContact.objects.filter(meeting=meeting)
    if meeting.abstract_request:
        abstracts = AbstractSubmission.objects.filter(abstract_request=meeting.abstract_request,status='approved-op')
    else:
        abstracts = AbstractSubmission.objects.none()

    form = SessionPresentationForm()
    form.fields['speaker'].queryset = contacts
    form.fields['speaker_2'].queryset = contacts
    form.fields['speaker_3'].queryset = contacts
    form.fields['speaker_4'].queryset = contacts
    form.fields['speaker_5'].queryset = contacts
    #form.fields['abstract_submission'].queryset = abstracts

    if request.POST:
        form = SessionPresentationForm(request.POST)
        form.fields['speaker'].queryset = contacts
        form.fields['speaker_2'].queryset = contacts
        form.fields['speaker_3'].queryset = contacts
        form.fields['speaker_4'].queryset = contacts
        form.fields['speaker_5'].queryset = contacts
        #form.fields['abstract_submission'].queryset = abstracts

        if form.is_valid():
            session_presentation = form.save(commit=False)
            session_presentation.meeting_session = session
            session_presentation.save()

            messages.success(request,'Presentation Has Been Created')

            return redirect('admin_meeting_information_session_presentations',meeting_id=meeting_id,session_id=session_id)

    return render(request,'admin/meeting_information/programme/presentations/add-presentation.html',{'meeting':meeting,'session':session,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_session_presentation(request,meeting_id,session_id,pres_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)
    presentation = get_object_or_404(SessionPresentation,id=pres_id)

    contacts = MeetingContact.objects.filter(meeting=meeting)
    if meeting.abstract_request:
        abstracts = AbstractSubmission.objects.filter(abstract_request=meeting.abstract_request,status='approved-op')
    else:
        abstracts = AbstractSubmission.objects.none()

    form = SessionPresentationForm(instance=presentation)
    form.fields['speaker'].queryset = contacts
    form.fields['speaker_2'].queryset = contacts
    form.fields['speaker_3'].queryset = contacts
    form.fields['speaker_4'].queryset = contacts
    form.fields['speaker_5'].queryset = contacts
    #form.fields['abstract_submission'].queryset = abstracts

    if request.POST:
        form = SessionPresentationForm(request.POST,instance=presentation)
        form.fields['speaker'].queryset = contacts
        form.fields['speaker_2'].queryset = contacts
        form.fields['speaker_3'].queryset = contacts
        form.fields['speaker_4'].queryset = contacts
        form.fields['speaker_5'].queryset = contacts
        #form.fields['abstract_submission'].queryset = abstracts
        if form.is_valid():
            session_presentation = form.save()

            messages.success(request,'Presentation Has Been Updated')

            return redirect('admin_meeting_information_session_presentations',meeting_id=meeting_id,session_id=session_id)

    return render(request,'admin/meeting_information/programme/presentations/edit-presentation.html',{'meeting':meeting,'session':session,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_session_presentation(request,meeting_id,session_id,pres_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)
    presentation = get_object_or_404(SessionPresentation,id=pres_id)

    presentation.delete()
    messages.success(request,'Presentation Has Been Deleted')

    return redirect('admin_meeting_information_session_presentations',meeting_id=meeting_id,session_id=session_id)

@permission_required('admin_users.can_access_meeting_information')
def programme_options(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    days = SessionDay.objects.filter(meeting=meeting)
    session_types = SessionType.objects.all()
    subjects = SessionSubject.objects.all()
    locations = SessionLocation.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/options/options.html',{'meeting':meeting,'days':days,'session_types':session_types,'subjects':subjects,'locations':locations})

@permission_required('admin_users.can_access_meeting_information')
def add_day(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = SessionDayForm()

    if request.POST:
        form = SessionDayForm(request.POST)

        if form.is_valid():
            day = form.save(commit=False)
            day.meeting = meeting
            day.save()

            messages.success(request,'Day has been added')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/days/add-day.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_day(request,meeting_id,day_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    day = get_object_or_404(SessionDay,id=day_id)
    form = SessionDayForm(instance=day)

    if request.POST:
        form = SessionDayForm(request.POST,instance=day)

        if form.is_valid():
            form.save()

            messages.success(request,'Day has been updated')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/days/edit-day.html',{'meeting':meeting,'day':day,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_day(request,meeting_id,day_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    day = get_object_or_404(SessionDay,id=day_id)

    day.delete()
    messages.success(request,'Day has been deleted')
    return redirect('admin_meeting_information_options',meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def add_location(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = SessionLocationForm()

    if request.POST:
        form = SessionLocationForm(request.POST)

        if form.is_valid():
            location = form.save(commit=False)
            location.meeting = meeting
            location.save()

            messages.success(request,'Location has been added')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/locations/add-location.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_location(request,meeting_id,location_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    location = get_object_or_404(SessionLocation,id=location_id)
    form = SessionLocationForm(instance=location)

    if request.POST:
        form = SessionLocationForm(request.POST,instance=location)

        if form.is_valid():
            form.save()

            messages.success(request,'Location has been updated')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/locations/edit-location.html',{'meeting':meeting,'location':location,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_location(request,meeting_id,location_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    location = get_object_or_404(SessionLocation,id=location_id)

    location.delete()
    messages.success(request,'Location has been deleted')
    return redirect('admin_meeting_information_options',meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def add_session_type(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = SessionTypeForm()

    if request.POST:
        form = SessionTypeForm(request.POST)

        if form.is_valid():
            type = form.save()

            messages.success(request,'Session type has been added')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/types/add-type.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_session_type(request,meeting_id,type_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    type = get_object_or_404(SessionType,id=type_id)
    form = SessionTypeForm(instance=type)

    if request.POST:
        form = SessionTypeForm(request.POST,instance=type)

        if form.is_valid():
            form.save()

            messages.success(request,'Session type has been updated')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/types/edit-type.html',{'meeting':meeting,'type':type,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_session_type(request,meeting_id,type_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    type = get_object_or_404(SessionType,id=type_id)

    type.delete()
    messages.success(request,'Session Type has been deleted')
    return redirect('admin_meeting_information_options',meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def add_subject(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = SessionSubjectForm()

    if request.POST:
        form = SessionSubjectForm(request.POST)

        if form.is_valid():
            type = form.save()

            messages.success(request,'Session Subject has been added')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/subjects/add-subject.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_subject(request,meeting_id,subject_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    subject = get_object_or_404(SessionSubject,id=subject_id)
    form = SessionSubjectForm(instance=subject)

    if request.POST:
        form = SessionSubjectForm(request.POST,instance=subject)

        if form.is_valid():
            form.save()

            messages.success(request,'Session Subject has been updated')
            return redirect('admin_meeting_information_options',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/options/subjects/edit-subject.html',{'meeting':meeting,'subject':subject,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_subject(request,meeting_id,subject_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    subject = get_object_or_404(SessionSubject,id=subject_id)

    subject.delete()
    messages.success(request,'Session Subject has been deleted')
    return redirect('admin_meeting_information_options',meeting_id=meeting.id)

##################################################
####################contacts######################
##################################################

@permission_required('admin_users.can_access_meeting_information')
def contacts(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    contacts = MeetingContact.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/contacts/contacts.html',{'meeting':meeting,'contacts':contacts})

@permission_required('admin_users.can_access_meeting_information')
def add_contact(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingContactForm()

    if request.POST:
        form = MeetingContactForm(request.POST)
        if form.is_valid():
            if form.cleaned_data['member']:
                member = form.cleaned_data['member']

                try:
                    contact = MeetingContact.objects.get(email_address=member.user.email,meeting=meeting)
                    messages.error(request,'A contact already exists for this member.')
                except MeetingContact.DoesNotExist:
                    contact = MeetingContact(member=member,meeting=meeting,title=member.title,first_name=member.given_name,last_name=member.surname,email_address=member.user.email,institution=member.hospital,bio=form.cleaned_data['bio'])
                    contact.save()

                    messages.success(request,'Contact has been created from member')
                    return redirect('admin_meeting_information_contacts',meeting_id=meeting_id)

            elif form.cleaned_data['title'] and form.cleaned_data['first_name'] and form.cleaned_data['last_name']:
                contact = form.save(commit=False)
                contact.meeting = meeting
                contact.save()

                messages.success(request,'Contact has been created from details entered')
                return redirect('admin_meeting_information_contacts',meeting_id=meeting_id)

            else:
                messages.error(request,'Please either enter contacts details or select a member')

    return render(request,'admin/meeting_information/contacts/add-contact.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def add_member_contacts(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingMemberContactForm()

    if request.POST:
        form = MeetingMemberContactForm(request.POST)
        if form.is_valid():
            members = form.cleaned_data['members']
            for member in members:
                try:
                    contact = MeetingContact.objects.get(email_address=member.user.email,meeting=meeting)
                except MeetingContact.DoesNotExist:
                    contact = MeetingContact(member=member,meeting=meeting,title=member.title,first_name=member.given_name,last_name=member.surname,email_address=member.user.email,institution=member.hospital)
                    contact.save()

            messages.success(request,'Members have been added')
            return redirect('admin_meeting_information_contacts',meeting_id=meeting_id)

    return render(request,'admin/meeting_information/contacts/add-member-contacts.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def import_contacts(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    if meeting.abstract_request:
        abstracts = AbstractSubmission.objects.filter(Q(status='approved-op') | Q(status='approved-poster'), abstract_request=meeting.abstract_request)

        for abstract in abstracts:
            try:
                contact = MeetingContact.objects.get(email_address=abstract.email,meeting=meeting)
            except:
                meeting_contact = MeetingContact(meeting=meeting,title=abstract.title,first_name=abstract.given_name,last_name=abstract.surname,email_address=abstract.email)
                try:
                    meeting_contact.member = abstract.member
                except:
                    pass
                institution = abstract.get_institution()
                if institution:
                    meeting_contact.institution = institution
                else:
                    meeting_contact.institution = abstract.member.organisation
                meeting_contact.save()

        messages.success(request,'Contacts have been imported')
    else:
        messages.error(request,'Sorry, there are no contacts to import')

    return redirect('admin_meeting_information_contacts',meeting_id=meeting_id)

@permission_required('admin_users.can_access_meeting_information')
def edit_contact(request,meeting_id,contact_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    contact = get_object_or_404(MeetingContact,id=contact_id,meeting=meeting)

    form = MeetingContactForm(instance=contact)

    if request.POST:
        form = MeetingContactForm(request.POST,instance=contact)
        if form.is_valid():
            form.save()

            messages.success(request,'Contact has been updated')
            return redirect('admin_meeting_information_contacts',meeting_id=meeting_id)

    return render(request,'admin/meeting_information/contacts/edit-contact.html',{'meeting':meeting,'contact':contact,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_contact(request,meeting_id,contact_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    contact = get_object_or_404(MeetingContact,id=contact_id,meeting=meeting)

    contact.delete()
    messages.success(request,'Contact has been deleted')

    return redirect('admin_meeting_information_contacts',meeting_id=meeting_id)


@permission_required('admin_users.can_access_meeting_information')
def emailers(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    emailers = MeetingEmailer.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/emailers/emailers.html',{'meeting':meeting,'emailers':emailers})

@permission_required('admin_users.can_access_meeting_information')
def add_emailer(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingEmailerForm()

    if request.POST:
        form = MeetingEmailerForm(request.POST)
        if form.is_valid():

            emailer = form.save(commit=False)
            emailer.meeting = meeting
            emailer.save()

            messages.success(request, "Emailer has been created")
            return redirect('admin_meeting_information_emailers',meeting_id=meeting_id)

    return render(request, 'admin/meeting_information/emailers/add-emailer.html', {'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_emailer(request,meeting_id,emailer_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    emailer = get_object_or_404(MeetingEmailer,id=emailer_id,meeting=meeting)

    form = MeetingEmailerForm(instance=emailer)

    if request.POST:
        form = MeetingEmailerForm(request.POST,instance=emailer)
        if form.is_valid():
            form.save()

            messages.success(request,'Emailer has been updated')
            return redirect('admin_meeting_information_emailers',meeting_id=meeting_id)

    return render(request,'admin/meeting_information/emailers/edit-emailer.html',{'meeting':meeting,'emailer':emailer,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_emailer(request,meeting_id,emailer_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    emailer = get_object_or_404(MeetingEmailer,id=emailer_id,meeting=meeting)

    emailer.delete()
    messages.success(request,'Emailer has been deleted')

    return redirect('admin_meeting_information_emailers',meeting_id=meeting_id)


#abstract sessions
@permission_required('admin_users.can_access_meeting_information')
def posters(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    posters = MeetingPoster.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/posters/posters.html',{'meeting':meeting,'posters':posters})

@permission_required('admin_users.can_access_meeting_information')
def import_posters(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    abstracts = AbstractSubmission.objects.filter(abstract_request=meeting.abstract_request,status='approved-poster')
    if abstracts:
        for abstract in abstracts:
            try:
                poster = MeetingPoster.objects.get(meeting=meeting,abstract_submission=abstract)
            except MeetingPoster.DoesNotExist:
                #create new
                poster = MeetingPoster(meeting=meeting,title=abstract.presentation_title,abstract_submission=abstract,abstract=abstract.abstract)

                if abstract.content_type.slug == 'educational-poster':
                    poster.poster_type = 'educational'
                else:
                    poster.poster_type = 'scientific'

                if abstract.poster_file:
                    poster.poster_file = abstract.poster_file

                try:
                    meeting_contact = MeetingContact.objects.get(email_address=abstract.email,meeting=meeting)
                    poster.contact = meeting_contact
                except MeetingContact.DoesNotExist:
                    meeting_contact = MeetingContact(meeting=meeting,title=abstract.title,first_name=abstract.given_name,last_name=abstract.surname,email_address=abstract.email)
                    try:
                        meeting_contact.member = abstract.member
                    except:
                        pass
                    institution = abstract.get_institution()
                    if institution:
                        meeting_contact.institution = institution
                    else:
                        meeting_contact.institution = abstract.member.organisation

                    meeting_contact.save()

                    poster.contact = meeting_contact

                poster.save()

        messages.success(request,'Posters have been added')
    else:
        messages.warning(request,'There are no posters to add')

    return redirect('admin_meeting_information_posters',meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def add_poster(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    if meeting.abstract_request:
        abstracts = AbstractSubmission.objects.filter(abstract_request=meeting.abstract_request,status='approved-poster')
    else:
        abstracts = AbstractSubmission.objects.none()
    form = MeetingPosterForm()
    form.fields['abstract_submission'].queryset = abstracts

    if request.POST:
        form = MeetingPosterForm(request.POST,request.FILES)
        form.fields['abstract_submission'].queryset = abstracts

        if form.is_valid():
            poster = form.save(commit=False)
            poster.meeting = meeting
            poster.save()

            if not poster.poster_file and poster.abstract_submission.poster_file:
                poster.poster_file = poster.abstract_submission.poster_file
                poster.save()

            messages.success(request,'Poster Created')

            return redirect('admin_meeting_information_posters',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/posters/add-poster.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_poster(request,meeting_id,poster_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    poster = get_object_or_404(MeetingPoster,id=poster_id)

    if meeting.abstract_request:
        abstracts = AbstractSubmission.objects.filter(abstract_request=meeting.abstract_request,status='approved-poster')
    else:
        abstracts = AbstractSubmission.objects.none()

    form = MeetingPosterForm(instance=poster)
    form.fields['abstract_submission'].queryset = abstracts

    if request.POST:
        form = MeetingPosterForm(request.POST,request.FILES,instance=poster)

        form.fields['abstract_submission'].queryset = abstracts

        if form.is_valid():
            form.save()
            messages.success(request,'Poster Updated')

            return redirect('admin_meeting_information_posters',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/posters/edit-poster.html',{'meeting':meeting,'poster':poster,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_poster(request,meeting_id,poster_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    poster = get_object_or_404(MeetingPoster,id=poster_id)

    poster.delete()
    messages.success(request,'Poster Deleted')

    return redirect('admin_meeting_information_posters',meeting_id=meeting.id)

#exhibitors
@permission_required('admin_users.can_access_meeting_information')
def exhibitors(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    exhibitors = MeetingExhibitor.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/exhibitors/exhibitors.html',{'meeting':meeting,'exhibitors':exhibitors})

@permission_required('admin_users.can_access_meeting_information')
def add_exhibitor(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    form = MeetingExhibitorForm()
    form.fields['floor_plan'].queryset = MeetingFloorPlan.objects.filter(meeting=meeting)

    if request.POST:
        form = MeetingExhibitorForm(request.POST,request.FILES)
        form.fields['floor_plan'].queryset = MeetingFloorPlan.objects.filter(meeting=meeting)

        if form.is_valid():
            exhibitor = form.save(commit=False)
            exhibitor.meeting = meeting
            exhibitor.save()

            messages.success(request,'Exhibitor Created')

            return redirect('admin_meeting_information_exhibitors',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/exhibitors/add-exhibitor.html',{'meeting':meeting,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def edit_exhibitor(request,meeting_id,exhibitor_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    exhibitor = get_object_or_404(MeetingExhibitor,id=exhibitor_id)

    form = MeetingExhibitorForm(instance=exhibitor)
    form.fields['floor_plan'].queryset = MeetingFloorPlan.objects.filter(meeting=meeting)

    if request.POST:
        form = MeetingExhibitorForm(request.POST,request.FILES,instance=exhibitor)
        form.fields['floor_plan'].queryset = MeetingFloorPlan.objects.filter(meeting=meeting)

        if form.is_valid():
            form.save()
            messages.success(request,'Exhibitor Updated')

            return redirect('admin_meeting_information_exhibitors',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/exhibitors/edit-exhibitor.html',{'meeting':meeting,'exhibitor':exhibitor,'form':form})

@permission_required('admin_users.can_access_meeting_information')
def delete_exhibitor(request,meeting_id,exhibitor_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    exhibitor = get_object_or_404(MeetingExhibitor,id=exhibitor_id)

    exhibitor.delete()
    messages.success(request,'Exhibitor Deleted')

    return redirect('admin_meeting_information_exhibitors',meeting_id=meeting.id)

#################voting#################
@permission_required('admin_users.can_access_meeting_information')
def votes(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    votes = MeetingVote.objects.filter(meeting=meeting)

    return render(request,'admin/meeting_information/votes/votes.html',{'meeting':meeting,'votes': votes})

@permission_required('admin_users.can_access_meeting_information')
def add_vote(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    form = MeetingVoteForm()

    if request.POST:
        form = MeetingVoteForm(request.POST,request.FILES)
        if form.is_valid():
            vote = form.save(commit=False)
            vote.meeting = meeting
            vote.save()

            messages.success(request,'Vote Created')
            return redirect('admin_meeting_information_votes',meeting_id=meeting.id)
    return render(request,'admin/meeting_information/votes/add-vote.html', {'meeting':meeting,'form': form})

@permission_required('admin_users.can_access_meeting_information')
def edit_vote(request, meeting_id, vote_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    vote = get_object_or_404(MeetingVote,id=vote_id)
    form = MeetingVoteForm(instance=vote)
    if request.POST:
        form = MeetingVoteForm(request.POST,request.FILES,instance=vote)

        if form.is_valid():
            form.save()
            messages.success(request,'Link Updated')

            return redirect('admin_meeting_information_votes',meeting_id=meeting.id)

    return render(request,'admin/meeting_information/votes/edit-vote.html', {'meeting':meeting,'form': form, 'vote': vote})

@permission_required('admin_users.can_access_meeting_information')
def delete_vote(request,meeting_id,vote_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    vote = get_object_or_404(MeetingVote,id=vote_id)
    vote.delete()
    messages.success(request,'Vote Deleted')

    return redirect('admin_meeting_information_votes',meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def vote_questions(request,meeting_id,vote_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    vote = get_object_or_404(MeetingVote,id=vote_id)

    VQFS = modelformset_factory(MeetingVoteQuestion,fields=('name','options'),can_delete=True,extra=0)
    question_formset = VQFS(queryset=MeetingVoteQuestion.objects.filter(vote=vote))

    if request.POST:
        question_formset = VQFS(request.POST,queryset=MeetingVoteQuestion.objects.filter(vote=vote))

        errors = []
        order = 1

        for form in question_formset:
            if form.is_valid():
                if 'name' in form.cleaned_data:
                    question = form.save(commit=False)
                    question.vote = vote
                    question.order = order
                    question.save()
            else:
                errors.append(form.errors)
            order += 1
        if not errors:

            for form in question_formset.deleted_forms:
                form.instance.delete()

            messages.success(request,'Questions have been saved')
            return redirect('admin_meeting_information_votes',meeting_id=meeting.id)

        else :
            messages.error(request,'Could not save your questions, please see errors below.')
            for error in errors:
                messages.error(request,error)

    return render(request,'admin/meeting_information/votes/questions.html',{'meeting':meeting,'vote':vote,'question_formset':question_formset})

@permission_required('admin_users.can_access_meeting_information')
def vote_submissions(request,meeting_id,vote_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    vote = get_object_or_404(MeetingVote,id=vote_id)

    questions = MeetingVoteQuestion.objects.filter(vote=vote)

    return render(request, 'admin/meeting_information/votes/submissions.html', {'meeting':meeting, 'vote':vote, 'questions':questions})


@permission_required('admin_users.can_access_meeting_information')
def send_vote_reminder(request, meeting_id, vote_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    vote = get_object_or_404(MeetingVote,id=vote_id)

    form = MeetingVoteReminderForm()

    if request.POST:
        form = MeetingVoteReminderForm(request.POST)

        if form.is_valid():
            vr = form.save(commit=False)
            vr.vote = vote
            vr.save()

        vote_reminder(vr=vr)

        messages.success(request, 'Your vote has been sent')

        return redirect('admin_meeting_information_votes', meeting_id=meeting_id)
    
    return render(request, 'admin/meeting_information/votes/reminder.html', {'meeting': meeting, 'vote':vote, 'form':form})


@permission_required('admin_users.can_access_meeting_information')
def evaluation(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    sessions = ProgrammeSession.objects.filter(evaluation_enabled=True,meeting=meeting)
    posters = MeetingPoster.objects.filter(meeting=meeting)

    return render(request, 'admin/meeting_information/evaluation/evaluation.html',{'meeting':meeting, 'sessions':sessions, 'posters':posters})

@permission_required('admin_users.can_access_meeting_information')
def evaluation_evaluators(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    evaluator_form = MeetingEvaluatorForm(instance=meeting)

    if request.POST:
        evaluator_form = MeetingEvaluatorForm(request.POST,instance=meeting)
        if evaluator_form.is_valid():
            evaluator_form.save()

            messages.success(request,'Evaluators have been saved')
            return redirect('admin_meeting_information_evaluation', meeting_id=meeting.id)

    return render(request, 'admin/meeting_information/evaluation/evaluators.html', {'meeting':meeting, 'evaluator_form':evaluator_form})


@permission_required('admin_users.can_access_meeting_information')
def add_all_evaluators(request, meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    booking_member_ids = MeetingBooking.objects.filter(meeting=meeting.meeting_booking, complete=True).values_list('member_id', flat=True)
    members = Member.objects.filter(id__in=booking_member_ids)

    for member in members:
        meeting.evaluators.add(member)
    meeting.save()

    messages.success(request,'Evaluators have been saved')
    return redirect('admin_meeting_information_evaluation_evaluators', meeting_id=meeting.id)

@permission_required('admin_users.can_access_meeting_information')
def evaluation_session(request,meeting_id,session_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)
    submissions = SessionEvaluationSubmission.objects.filter(session=session)

    return render(request,'admin/meeting_information/evaluation/session.html',{'meeting':meeting, 'session':session, 'submissions':submissions })

@permission_required('admin_users.can_access_meeting_information')
def download_evaluation_submissions(request,meeting_id,session_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)
    submissions = SessionEvaluationSubmission.objects.filter(session=session)

    before_qs = SessionEvaluationQuestion.objects.filter(meeting=meeting,after=False)
    after_qs = SessionEvaluationQuestion.objects.filter(meeting=meeting,after=True)
    presentations = SessionPresentation.objects.filter(meeting_session=session)
    presentation_questions = PresentationEvaluationQuestion.objects.filter(meeting=meeting)

    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=scoring-submissions-%s.csv" % (session.id)

    import csv
    writer = csv.writer(response)

    fields = ['Name','Email Address']
    for q in before_qs:
        fields.append(q.name)
    for pres in presentations:
        for question in presentation_questions:
            fields.append("%s - %s" % (pres.title, question.name))
    for q in after_qs:
        fields.append(q.name)

    writer.writerow(fields)

    for submission in submissions:
        submission_fields = [submission.member,submission.member.user.email]
        for answer in submission.get_answers():
            submission_fields.append(answer.answer)

        writer.writerow([unicode(s).encode("utf-8") for s in submission_fields])

    return response


@permission_required('admin_users.can_access_meeting_information')
def download_sessions(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    submissions = SessionEvaluationSubmission.objects.filter(session__meeting=meeting).order_by('session')

    before_qs = SessionEvaluationQuestion.objects.filter(meeting=meeting,after=False)
    after_qs = SessionEvaluationQuestion.objects.filter(meeting=meeting,after=True)
    presentations = SessionPresentation.objects.filter(meeting_session__meeting=meeting)
    presentation_questions = PresentationEvaluationQuestion.objects.filter(meeting=meeting)

    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=session-submissions.csv"

    import csv
    writer = csv.writer(response)

    fields = ['Session', 'Name','Email Address']
    for q in before_qs:
        fields.append(q.name)
    for q in after_qs:
        fields.append(q.name)

    writer.writerow(fields)

    for submission in submissions:
        submission_fields = [submission.session, submission.member,submission.member.user.email]
        for answer in submission.get_non_pres_answers():
            submission_fields.append(answer.answer)

        writer.writerow([unicode(s).encode("utf-8") for s in submission_fields])

    return response


@permission_required('admin_users.can_access_meeting_information')
def download_presentations(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    presentations = SessionPresentation.objects.filter(meeting_session__meeting=meeting)
    presentation_questions = PresentationEvaluationQuestion.objects.filter(meeting=meeting)

    submissions = SessionEvaluationSubmission.objects.filter(session__meeting=meeting)

    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=presentation-scores.csv"

    import csv
    writer = csv.writer(response)

    fields = ['Session', 'Presentation', 'Name', 'Email Address']

    for question in presentation_questions:
        fields.append(question.name)

    writer.writerow(fields)

    print submissions

    for presentation in presentations:
        for submission in submissions:
            submission_fields = [presentation.meeting_session, presentation.title, submission.member, submission.member.user.email]
            answers = submission.get_presentation_answers(presentation)
            if answers:
                for answer in answers:
                    submission_fields.append(answer.answer)
                writer.writerow([unicode(s).encode("utf-8") for s in submission_fields])

    return response

@permission_required('admin_users.can_access_meeting_information')
def download_average_presentation_scores(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    presentations = SessionPresentation.objects.filter(meeting_session__meeting=meeting)

    questions = PresentationEvaluationQuestion.objects.filter(meeting=meeting, type='scale')
    yesno_questions =  PresentationEvaluationQuestion.objects.filter(meeting=meeting, type='yes-no')

    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=presentation-average-scores.csv"

    import csv
    writer = csv.writer(response)

    fields = ['Session', 'Presentation', 'Submission Count']
    for q in questions:
        fields.append(q.name)
    for q in yesno_questions:
        fields.append("%s (Yes Count)" % (q.name))

    print yesno_questions

    writer.writerow(fields)

    submissions = SessionEvaluationSubmission.objects.filter(session__meeting=meeting)

    for presentation in presentations:

        submissions = SessionEvaluationSubmission.objects.filter(session=presentation.meeting_session)
        
        if submissions:

            submission_fields = [presentation.meeting_session, presentation.title, submissions.count()]

            for question in questions:
                score = presentation.get_average_for_question(question)
                submission_fields.append(score)

            for q in yesno_questions:
                yesses = presentation.get_yesses_for_question(q)
                submission_fields.append(yesses)

            writer.writerow([unicode(s).encode("utf-8") for s in submission_fields])

    return response

@permission_required('admin_users.can_access_meeting_information')
def evaluation_session_view_submission(request,meeting_id,session_id,submission_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)
    submission = get_object_or_404(SessionEvaluationSubmission,id=submission_id)

    answers = submission.get_answers()

    return render(request,'admin/meeting_information/evaluation/submission.html',{'meeting':meeting, 'session':session, 'submission':submission, 'answers':answers })

@permission_required('admin_users.can_access_meeting_information')
def evaluation_session_delete_submission(request,meeting_id,session_id,submission_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    session = get_object_or_404(ProgrammeSession,id=session_id)
    submission = get_object_or_404(SessionEvaluationSubmission,id=submission_id)

    submission.delete()

    messages.success(request,'Submission Deleted')

    return redirect('admin_meeting_information_evaluation_session', meeting_id=meeting.id, session_id=session.id)

@permission_required('admin_users.can_access_meeting_information')
def evaluation_poster(request,meeting_id,poster_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    poster = get_object_or_404(MeetingPoster,id=poster_id)
    submissions = PosterEvaluationSubmission.objects.filter(poster=poster)

    return render(request,'admin/meeting_information/evaluation/poster.html',{'meeting':meeting, 'poster':poster, 'submissions':submissions })

@permission_required('admin_users.can_access_meeting_information')
def evaluation_poster_download(request,meeting_id,poster_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    poster = get_object_or_404(MeetingPoster,id=poster_id)
    submissions = PosterEvaluationSubmission.objects.filter(poster=poster)

    questions = PosterEvaluationQuestion.objects.filter(meeting=meeting)

    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=poster-submissions-%s.csv" % (poster.id)

    import csv
    writer = csv.writer(response)

    fields = ['Name','Email Address']
    for q in questions:
        fields.append(q.name)

    writer.writerow(fields)

    for submission in submissions:
        submission_fields = [submission.member,submission.member.user.email]
        for answer in submission.get_answers():
            submission_fields.append(answer.answer)

        writer.writerow([unicode(s).encode("utf-8") for s in submission_fields])

    return response


@permission_required('admin_users.can_access_meeting_information')
def download_poster_scores(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    submissions = PosterEvaluationSubmission.objects.filter(poster__meeting=meeting).order_by('poster')

    questions = PosterEvaluationQuestion.objects.filter(meeting=meeting)

    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=poster-scores-average.csv"

    import csv
    writer = csv.writer(response)

    fields = ['Poster', 'Name','Email Address']
    for q in questions:
        fields.append(q.name)

    writer.writerow(fields)

    for submission in submissions:
        submission_fields = [submission.poster, submission.member,submission.member.user.email]
        for answer in submission.get_answers():
            submission_fields.append(answer.answer)

        writer.writerow([unicode(s).encode("utf-8") for s in submission_fields])

    return response


@permission_required('admin_users.can_access_meeting_information')
def download_average_poster_scores(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    posters = MeetingPoster.objects.filter(meeting=meeting)

    questions = PosterEvaluationQuestion.objects.filter(meeting=meeting, type='scale')
    yesno_questions =  PosterEvaluationQuestion.objects.filter(meeting=meeting, type='yes-no')

    # Create the HttpResponse object with the appropriate CSV header.
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=poster-scores.csv"

    import csv
    writer = csv.writer(response)

    fields = ['Poster', 'Submission Count']
    for q in questions:
        fields.append(q.name)
    for q in yesno_questions:
        fields.append("%s (Yes Count)" % (q.name))

    writer.writerow(fields)

    for poster in posters:
        submissions = PosterEvaluationSubmission.objects.filter(poster=poster).order_by('poster')

        submission_fields = [poster, submissions.count()]

        for question in questions:
            score = question.get_average_for_poster(poster)
            submission_fields.append(score)

        for q in yesno_questions:
            yesses = q.get_yesses_for_poster(poster)
            submission_fields.append(yesses)

        writer.writerow([unicode(s).encode("utf-8") for s in submission_fields])

    return response


@permission_required('admin_users.can_access_meeting_information')
def evaluation_poster_view_submission(request,meeting_id,poster_id,submission_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    poster = get_object_or_404(MeetingPoster,id=poster_id)
    submission = get_object_or_404(PosterEvaluationSubmission,id=submission_id)

    answers = submission.get_answers()

    return render(request,'admin/meeting_information/evaluation/poster-submission.html',{'meeting':meeting, 'poster':poster, 'submission':submission, 'answers':answers })

@permission_required('admin_users.can_access_meeting_information')
def evaluation_poster_delete_submission(request,meeting_id,poster_id,submission_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)
    poster = get_object_or_404(MeetingPoster,id=poster_id)
    submission = get_object_or_404(PosterEvaluationSubmission,id=submission_id)

    submission.delete()

    messages.success(request,'Submission Deleted')

    return redirect('admin_meeting_information_evaluation_poster', meeting_id=meeting.id, poster_id=poster.id)


@permission_required('admin_users.can_access_meeting_information')
def eval_questions(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    MEQFS = modelformset_factory(SessionEvaluationQuestion,fields=('name','after','type','lower_scale','upper_scale'),can_delete=True,extra=0)
    question_formset = MEQFS(queryset=SessionEvaluationQuestion.objects.filter(meeting=meeting))

    if request.POST:
        question_formset = MEQFS(request.POST,queryset=SessionEvaluationQuestion.objects.filter(meeting=meeting))

        errors = []
        order = 1

        for form in question_formset:
            if form.is_valid():
                if 'name' in form.cleaned_data:
                    question = form.save(commit=False)
                    question.meeting = meeting
                    question.order = order
                    question.save()
            else:
                errors.append(form.errors)
            order += 1
        if not errors:

            for form in question_formset.deleted_forms:
                form.instance.delete()

            messages.success(request,'Questions have been saved')
            return redirect('admin_meeting_information_evaluation', meeting_id=meeting.id)

        else :
            messages.error(request,'Could not save your questions, please see errors below.')
            for error in errors:
                messages.error(request,error)

    return render(request,'admin/meeting_information/evaluation/questions.html',{'meeting':meeting, 'question_formset':question_formset})

@permission_required('admin_users.can_access_meeting_information')
def eval_presentation_questions(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    MEQFS = modelformset_factory(PresentationEvaluationQuestion,fields=('name','type','lower_scale','upper_scale'),can_delete=True,extra=0)
    question_formset = MEQFS(queryset=PresentationEvaluationQuestion.objects.filter(meeting=meeting))

    if request.POST:
        question_formset = MEQFS(request.POST,queryset=PresentationEvaluationQuestion.objects.filter(meeting=meeting))

        errors = []
        order = 1

        for form in question_formset:
            if form.is_valid():
                if 'name' in form.cleaned_data:
                    question = form.save(commit=False)
                    question.meeting = meeting
                    question.order = order
                    question.save()
            else:
                errors.append(form.errors)
            order += 1
        if not errors:

            for form in question_formset.deleted_forms:
                form.instance.delete()

            messages.success(request,'Questions have been saved')
            return redirect('admin_meeting_information_evaluation', meeting_id=meeting.id)

        else :
            messages.error(request,'Could not save your questions, please see errors below.')
            for error in errors:
                messages.error(request,error)

    return render(request,'admin/meeting_information/evaluation/presentation-questions.html',{'meeting':meeting, 'question_formset':question_formset})

@permission_required('admin_users.can_access_meeting_information')
def eval_poster_questions(request,meeting_id):

    meeting = get_object_or_404(MeetingInformation,id=meeting_id)

    MEQFS = modelformset_factory(PosterEvaluationQuestion,fields=('name','type','lower_scale','upper_scale'),can_delete=True,extra=0)
    question_formset = MEQFS(queryset=PosterEvaluationQuestion.objects.filter(meeting=meeting))

    if request.POST:
        question_formset = MEQFS(request.POST,queryset=PosterEvaluationQuestion.objects.filter(meeting=meeting))

        errors = []
        order = 1

        for form in question_formset:
            if form.is_valid():
                if 'name' in form.cleaned_data:
                    question = form.save(commit=False)
                    question.meeting = meeting
                    question.order = order
                    question.save()
            else:
                errors.append(form.errors)
            order += 1
        if not errors:

            for form in question_formset.deleted_forms:
                form.instance.delete()

            messages.success(request,'Questions have been saved')
            return redirect('admin_meeting_information_evaluation', meeting_id=meeting.id)

        else :
            messages.error(request,'Could not save your questions, please see errors below.')
            for error in errors:
                messages.error(request,error)

    return render(request,'admin/meeting_information/evaluation/poster-questions.html',{'meeting':meeting, 'question_formset':question_formset})
