from django.shortcuts import render,redirect,get_object_or_404
from django.contrib import messages
from django.http import Http404, HttpResponse, JsonResponse
from django.core.urlresolvers import reverse
from django.contrib.auth.models import User
from django.conf import settings
from django.core.mail import EmailMessage
from django.db.models import Q, F

from datetime import datetime, date, time

from models import *
from forms import *
from emails import marketing_email

from decorators import meeting_members_only
from modules.members.models import Member
from modules.meeting_booking.models import Meeting, MeetingBooking

from modules.core.forms import LoginForm, ForgotPasswordForm,ChangePasswordForm
from modules.core.functions import random_string_unique
from modules.core.models import UserProfile

def home(request):

    if request.user.is_staff:
        meetings = MeetingInformation.objects.all()
    else:
        meetings = MeetingInformation.objects.filter(enabled=True)

    if meetings.count() == 1:
        meeting = meetings[0]
        return redirect('meeting_home',meeting_slug=meeting.slug)

    return render(request,'public/meeting_information/home.html',{'meetings':meetings})

@meeting_members_only
def meeting_home(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    home_links = MeetingLink.objects.filter(meeting=meeting,homepage=True)

    member_emailer = False
    if request.user.is_authenticated():
        try:
            member = Member.objects.get(user=request.user)
            try:
                member_emailer = MeetingEmailer.objects.get(member=member, meeting=meeting)
            except MeetingEmailer.DoesNotExist:
                pass
        except Member.DoesNotExist:
            pass

    if not member_emailer:
        home_links = home_links.exclude(link_type='message')

    show_splash = False
    if meeting.splash_enabled:
        if not request.session.get('splash_seen_%s' % (meeting.id)):
            request.session['splash_seen_%s' % (meeting.id)] = True
            show_splash = True

    return render(request,'public/meeting_information/meeting_home.html',{'home_links':home_links, 'show_splash': show_splash })


def meeting_login(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    from django.contrib.auth.models import User
    from django.contrib.auth import authenticate, login

    login_form = LoginForm()

    if request.POST:
        login_form = LoginForm(request.POST)

        if login_form.is_valid():

            email_address = login_form.cleaned_data['email_address']
            password = login_form.cleaned_data['password']

            try:
                user = User.objects.get(email=email_address)
                user = authenticate(username=user.username, password=password)
                #raise Exception(user.username)
                if user is not None:
                    if user.is_active:
                        try:
                            member = Member.objects.get(user=user)
                            login(request,user)

                            if request.GET.get('next'):
                                return redirect(request.GET['next'])
                            else:
                                return redirect('meeting_home',meeting_slug=meeting.slug)

                        except Member.DoesNotExist:

                            messages.error(request, 'Sorry, could not find a member account matching those details.')
                    else:
                        messages.error(request, "Sorry, your account is disabled.")
                else:
                    messages.error(request, "Your username or password is incorrect")
            except:
                messages.error(request, "Sorry, your username or password is incorrect - please try again.")

        else:
            messages.error(request,"Please ensure you have entered your details correctly.")

    return render(request,'public/meeting_information/login.html',{'form':login_form})


def meeting_logout(request,meeting_slug):

    from django.contrib.auth import logout

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    logout(request)

    return render(request,'public/meeting_information/logout.html',{})

def meeting_forgot_password(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    form = ForgotPasswordForm()
    user_error = False
    member = False

    if request.POST:
        form = ForgotPasswordForm(request.POST)

        try:
            member = Member.objects.get(user__email=request.POST['email'],user__is_active=True)
        except Member.DoesNotExist:
            member = False
            messages.error(request,'Sorry a user with that email address could not be found.')

        if member:

            N = 20
            forgot_key = random_string_unique(20, UserProfile, 'forgot_key')
            user_profile = member.user.userprofile
            user_profile.forgot_key = forgot_key
            user_profile.save()

            message = """
<p>A password reset form has been filled in for your account</p>

<p>Please click the following link to reset your password: -

<p><a href="http://%s/%s/forgot-password/%s/">Reset your password</a></p>

<p>This is an automatic email generated by the PSGBI website.</p>""" % (request.META['HTTP_HOST'],meeting.slug,forgot_key)

            msg = EmailMessage('PSGBI - Forgot Password', message, settings.CONTACT_EMAIL, [member.user.email])
            msg.content_subtype = "html"
            try:
                msg.send()
            except:
                pass
            return render(request,'public/meeting_information/forgot/forgot-confirm.html',{'email_reset':member.user.email})

    return render(request,'public/meeting_information/forgot/forgot-password.html',{'form':form})

def meeting_forgot_password_confirm(request,meeting_slug,forgot_key):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    try:
        user_profile = UserProfile.objects.get(forgot_key=forgot_key)
        user = user_profile.user
    except UserProfile.DoesNotExist:
        return render(request,'public/meeting_information/forgot/forgot-not-found.html',{})

    form = ChangePasswordForm()

    if request.POST:
        form = ChangePasswordForm(request.POST)

        if form.is_valid():
            try:

                user.set_password(form.cleaned_data['password1'])
                user.save()

                user_profile.forgot_key = ''
                user_profile.save()

                message = """
<p>Your password for the %s website has been reset.</p>

<p>You can log on at: -</p>

<p><a href="http://%s/%s/login/">http://%s/%s/login/</a></p>

<p>This is an automatic email generated by the %s.</p>""" % (settings.WEBSITE_NAME,request.META['HTTP_HOST'],meeting.slug,request.META['HTTP_HOST'],meeting.slug,settings.WEBSITE_NAME)

                msg = EmailMessage('%s - Forgot Password' % (settings.WEBSITE_NAME), message, settings.CONTACT_EMAIL, [user.email])
                msg.content_subtype = "html"

                try:
                    msg.send()
                except:
                    pass

                return render(request,'public/meeting_information/forgot/forgot-reset.html',{})

            except:
                messages.error(request,'Could not change password')

    return render(request,'public/meeting_information/forgot/forgot-reset-form.html',{'form':form})


@meeting_members_only
def programme(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    return render(request,'public/meeting_information/programme/programme.html',{})


@meeting_members_only
def programme_happening_now(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    current_date = date.today()
    current_time = datetime.now().time()
    sessions = ProgrammeSession.objects.filter(meeting=meeting,start_time__lte=current_time,end_time__gte=current_time,day__date=current_date)

    request.session['previous_url'] = reverse('programme_happening_now',args=[meeting.slug])

    return render(request,'public/meeting_information/programme/happening-now.html',{'sessions':sessions})

@meeting_members_only
def programme_by_day(request,meeting_slug,day_id=False,time_slug=False):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    days = SessionDay.objects.filter(meeting=meeting)
    if not days:
        return redirect('programme',meeting.slug)

    if day_id and time_slug:
        current_day = get_object_or_404(SessionDay,id=day_id,meeting=meeting)
        return render(request,'public/meeting_information/programme/by-day.html',{'days':days,'current_day':current_day,'current_time':time_slug})
    else:
        return render(request,'public/meeting_information/programme/by-day.html',{'days':days})

@meeting_members_only
def programme_load_day_time(request,meeting_slug,day_id,time_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    day = get_object_or_404(SessionDay,id=day_id,meeting=meeting)
    sessions = ProgrammeSession.objects.filter(meeting=meeting,time_of_day=time_slug,day=day)

    request.session['previous_url'] = reverse('programme_day_time',args=[meeting.slug,day.id,time_slug])

    return render(request,'public/meeting_information/programme/day-time.html',{'day':day,'time_of_day':time_slug,'sessions':sessions})

@meeting_members_only
def programme_by_type(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session_types = SessionType.objects.all()
    final_types = []
    for session_type in session_types:
        sessions = ProgrammeSession.objects.filter(session_type=session_type,meeting=meeting)
        if sessions:
            final_types.append(session_type)

    return render(request,'public/meeting_information/programme/by-type.html',{'session_types':final_types})

@meeting_members_only
def programme_type(request,meeting_slug,type_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session_type = get_object_or_404(SessionType,id=type_id)
    sessions = ProgrammeSession.objects.filter(session_type=session_type,meeting=meeting)

    request.session['previous_url'] = reverse('programme_type',args=[meeting.slug,session_type.id])

    return render(request,'public/meeting_information/programme/type.html',{'session_type':session_type,'sessions':sessions})

@meeting_members_only
def programme_by_subject(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session_subjects = SessionSubject.objects.all()
    final_subjects = []
    for subject in session_subjects:
        sessions = ProgrammeSession.objects.filter(primary_subject=subject,meeting=meeting)
        if sessions:
            final_subjects.append(subject)

    return render(request,'public/meeting_information/programme/by-subject.html',{'session_subjects':final_subjects})

@meeting_members_only
def programme_subject(request,meeting_slug,subject_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session_subject = get_object_or_404(SessionSubject,id=subject_id)
    sessions = ProgrammeSession.objects.filter(primary_subject=session_subject,meeting=meeting)

    request.session['previous_url'] = reverse('programme_subject',args=[meeting.slug,session_subject.id])

    return render(request,'public/meeting_information/programme/subject.html',{'session_subject':session_subject,'sessions':sessions})

@meeting_members_only
def programme_by_location(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session_locations = SessionLocation.objects.filter(meeting=meeting)

    return render(request,'public/meeting_information/programme/by-location.html',{'session_locations':session_locations})

@meeting_members_only
def programme_location(request,meeting_slug,location_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session_location = get_object_or_404(SessionLocation,id=location_id)
    sessions = ProgrammeSession.objects.filter(location=session_location,meeting=meeting)

    request.session['previous_url'] = reverse('programme_location',args=[meeting.slug,session_location.id])

    return render(request,'public/meeting_information/programme/location.html',{'session_location':session_location,'sessions':sessions})

@meeting_members_only
def programme_session(request,meeting_slug,session_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)
    presentations = session.get_presentations()

    user_session = False
    user_note = False
    if request.user.is_authenticated():
        try:
            member = Member.objects.get(user=request.user)
            try:
                user_session = UserSession.objects.get(member=member,session=session)
            except:
                pass
            try:
                user_note = UserNote.objects.get(member=member,session=session)
            except:
                pass
        except:
            pass

    if request.session.get('previous_url',False):
        previous_url = request.session['previous_url']
    else:
        previous_url = ''

    can_score = False

    if session.evaluation_enabled:
        if request.user.is_authenticated():
            try:
                member = Member.objects.get(user=request.user)
                if member in meeting.evaluators.all():
                    try:
                        session_evaluation_submission = SessionEvaluationSubmission.objects.get(session=session,member=member)
                    except SessionEvaluationSubmission.DoesNotExist:
                        can_score = True
            except:
                pass

    return render(request,'public/meeting_information/programme/session.html',{'session':session,'presentations':presentations,'user_session':user_session,'user_note':user_note,'previous_url':previous_url,'can_score':can_score})

@meeting_members_only
def programme_session_add(request, meeting_slug, session_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)

    try:
        member = Member.objects.get(user=request.user)
        try:
            user_session = UserSession.objects.get(member=member, session=session)
            user_session.delete()

            return JsonResponse({'message':'Session has been removed from your programme.','status':'removed'})

        except UserSession.DoesNotExist:
            #add user session
            user_session = UserSession(member=member, session=session)
            user_session.save()

            return JsonResponse({'message':'Session has been added to your programme.','status':'added'})

    except :
        return JsonResponse({'message':'Sorry, you need to be logged in to be able to add sessions to your programme. <a href="%s?next=%s">Login</a>.' % (reverse('meeting_login',args=[meeting_slug]),reverse('programme_session',args=[meeting_slug,session_id])), 'status':'unauthenticated'})

@meeting_members_only
def programme_session_note(request,meeting_slug,session_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,session=session)
    except:
        user_note = False

    return render(request,'public/meeting_information/programme/session_note.html',{'meeting':meeting,'session':session,'user_note':user_note})

@meeting_members_only
def programme_session_note_add(request,meeting_slug,session_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,session=session)
    except:
        user_note = False

    if request.POST:
        if user_note:
            user_note_form = UserNoteForm(request.POST,instance=user_note)
        else:
            user_note_form = UserNoteForm(request.POST)

        if user_note_form.is_valid():
            if user_note:
                user_note_form.save()
                return JsonResponse({'message':'Note has been updated.','status':'success'})
            else:
                user_note = user_note_form.save(commit=False)
                user_note.member = member
                user_note.session = session
                user_note.save()

                return JsonResponse({'message':'Note has been created.','status':'success'})

    return JsonResponse({'message':'Could not add note','status':'error'})

@meeting_members_only
def programme_session_evaluation(request,meeting_slug,session_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    before_questions = SessionEvaluationQuestion.objects.filter(meeting=meeting,after=False)
    after_questions = SessionEvaluationQuestion.objects.filter(meeting=meeting,after=True)
    presentations = session.get_presentations()
    presentation_questions = PresentationEvaluationQuestion.objects.filter(meeting=meeting)

    if not session.evaluation_enabled or member not in meeting.evaluators.all():
        return redirect('programme_session', meeting_slug=meeting.slug, session_id=session.id)

    try:
        session_evaluation_submission = SessionEvaluationSubmission.objects.get(session=session,member=member)
        return redirect('programme_session_evaluation_complete', meeting_slug=meeting.slug, session_id=session.id)
    except:
        pass

    if request.POST:

        print request.POST

        errors = False

        for question in before_questions:
            if not request.POST.get('question_%s' % (question.id)):
                errors = True
        for presentation in presentations:
            for question in presentation_questions:
                if not request.POST.get('presentation_%s_q_%s' % (presentation.id, question.id)):
                    errors = True
        for question in after_questions:
            if not request.POST.get('question_%s' % (question.id)):
                errors = True

        if not errors:
            submission = SessionEvaluationSubmission(session=session,member=member)
            submission.save()

            order = 1

            for question in before_questions:
                question_submission = SessionEvaluationSubmissionAnswer(evaluation_submission=submission,question_name=question.name, question=question,answer=request.POST['question_%s' % (question.id)],order=order)
                question_submission.save()
                order += 1

            for presentation in presentations:
                for question in presentation_questions:
                    question_submission = SessionEvaluationSubmissionAnswer(evaluation_submission=submission,presentation=presentation,question_name=question.name, answer=request.POST['presentation_%s_q_%s' % (presentation.id, question.id)],order=order)
                    question_submission.save()
                    order += 1

            for question in after_questions:
                question_submission = SessionEvaluationSubmissionAnswer(evaluation_submission=submission,question=question,question_name=question.name, answer=request.POST['question_%s' % (question.id)],order=order)
                question_submission.save()
                order += 1

            return JsonResponse({'message':'Evaluation has been submitted.','status':'success'})

        else:
            return JsonResponse({'message':'Sorry, could not submit this evaluation, please ensure you have answered all questions.','status':'error'})

    return render(request,'public/meeting_information/programme/session_evaluation.html',{'meeting':meeting, 'session':session, 'before_questions':before_questions, 'after_questions':after_questions, 'presentations':presentations, 'presentation_questions':presentation_questions})

@meeting_members_only
def programme_session_evaluation_complete(request,meeting_slug,session_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)

    return render(request,'public/meeting_information/programme/session_evaluation_complete.html',{'meeting':meeting, 'session':session})

def programme_presentation(request,meeting_slug,session_id,presentation_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)
    presentation = get_object_or_404(SessionPresentation,id=presentation_id,meeting_session=session)

    user_note = False
    if request.user.is_authenticated():
        try:
            member = Member.objects.get(user=request.user)
            try:
                user_note = UserNote.objects.get(member=member,presentation=presentation)
            except:
                pass
        except:
            pass

    return render(request,'public/meeting_information/programme/presentation.html',{'session':session,'presentation':presentation,'user_note':user_note})

@meeting_members_only
def programme_presentation_note(request,meeting_slug,session_id,presentation_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)
    presentation = get_object_or_404(SessionPresentation,id=presentation_id,meeting_session=session)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,presentation=presentation)
    except:
        user_note = False

    return render(request,'public/meeting_information/programme/presentation_note.html',{'meeting':meeting,'session':session,'presentation':presentation,'user_note':user_note})

@meeting_members_only
def programme_presentation_note_add(request,meeting_slug,session_id,presentation_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    session = get_object_or_404(ProgrammeSession,id=session_id,meeting=meeting)
    presentation = get_object_or_404(SessionPresentation,id=presentation_id,meeting_session=session)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,presentation=presentation)
    except:
        user_note = False

    if request.POST:
        if user_note:
            user_note_form = UserNoteForm(request.POST,instance=user_note)
        else:
            user_note_form = UserNoteForm(request.POST)

        if user_note_form.is_valid():
            if user_note:
                user_note_form.save()
                return JsonResponse({'message':'Note has been updated.','status':'success'})
            else:
                user_note = user_note_form.save(commit=False)
                user_note.member = member
                user_note.presentation = presentation
                user_note.save()

                return JsonResponse({'message':'Note has been created.','status':'success'})

    return JsonResponse({'message':'Could not add note','status':'error'})

@meeting_members_only
def floor_plans(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    floor_plans = MeetingFloorPlan.objects.filter(meeting=meeting)
    if floor_plans:
        return redirect('meeting_floorplan',meeting.slug,floor_plans[0].id)
    else:
        return redirect('meeting_home',meeting.slug)

@meeting_members_only
def floor_plan(request,meeting_slug,plan_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    floor_plans = MeetingFloorPlan.objects.filter(meeting=meeting)
    floor_plan = get_object_or_404(MeetingFloorPlan,meeting=meeting,id=plan_id)

    return render(request,'public/meeting_information/floor_plan.html',{'floor_plans':floor_plans,'floor_plan':floor_plan})

@meeting_members_only
def meeting_posters(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    educational_posters = MeetingPoster.objects.filter(meeting=meeting,poster_type='educational')
    scientific_posters = MeetingPoster.objects.filter(meeting=meeting,poster_type='scientific')

    return render(request,'public/meeting_information/posters/posters.html',{'educational_posters':educational_posters,'scientific_posters':scientific_posters})

@meeting_members_only
def meeting_poster(request,meeting_slug,poster_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    poster = get_object_or_404(MeetingPoster,id=poster_id,meeting=meeting)

    user_note = False
    if request.user.is_authenticated():
        try:
            member = Member.objects.get(user=request.user)
            try:
                user_note = UserNote.objects.get(member=member,poster=poster)
            except:
                pass
        except:
            pass

    can_score = False

    if request.user.is_authenticated():
        try:
            member = Member.objects.get(user=request.user)
            if member in meeting.evaluators.all():
                try:
                    poster_evaluation_submission = PosterEvaluationSubmission.objects.get(poster=poster,member=member)
                except PosterEvaluationSubmission.DoesNotExist:
                    can_score = True
        except:
            pass
        
    return render(request,'public/meeting_information/posters/poster-detail.html',{'poster':poster,'user_note':user_note, 'can_score':can_score})

@meeting_members_only
def meeting_poster_note(request,meeting_slug,poster_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    poster = get_object_or_404(MeetingPoster,id=poster_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,poster=poster)
    except:
        user_note = False

    return render(request,'public/meeting_information/posters/poster_note.html',{'poster':poster,'user_note':user_note})

@meeting_members_only
def meeting_poster_note_add(request,meeting_slug,poster_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    poster = get_object_or_404(MeetingPoster,id=poster_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,poster=poster)
    except:
        user_note = False

    if request.POST:
        if user_note:
            user_note_form = UserNoteForm(request.POST,instance=user_note)
        else:
            user_note_form = UserNoteForm(request.POST)

        if user_note_form.is_valid():
            if user_note:
                user_note_form.save()
                return JsonResponse({'message':'Note has been updated.','status':'success'})
            else:
                user_note = user_note_form.save(commit=False)
                user_note.member = member
                user_note.poster = poster
                user_note.save()

                return JsonResponse({'message':'Note has been created.','status':'success'})

    return JsonResponse({'message':'Could not add note','status':'error'})

@meeting_members_only
def meeting_poster_scoring(request,meeting_slug,poster_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    poster = get_object_or_404(MeetingPoster, id=poster_id, meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    if member not in meeting.evaluators.all():
        return redirect('meeting_poster', meeting_slug=meeting.slug, poster_id=poster.id)

    questions = PosterEvaluationQuestion.objects.filter(meeting=meeting)

    try:
        session_evaluation_submission = SessionEvaluationSubmission.objects.get(session=session,member=member)
        return redirect('programme_session_evaluation_complete', meeting_slug=meeting.slug, session_id=session.id)
    except:
        pass

    if request.POST:

        errors = False

        for question in questions:
            if not request.POST.get('question_%s' % (question.id)):
                errors = True

        if not errors:
            submission = PosterEvaluationSubmission(poster=poster, member=member)
            submission.save()

            order = 1

            for question in questions:
                question_submission = PosterEvaluationSubmissionAnswer(evaluation_submission=submission, question_name=question.name, question=question,answer=request.POST['question_%s' % (question.id)], order=order)
                question_submission.save()
                order += 1

            return JsonResponse({'message':'Evaluation has been submitted.','status':'success'})

        else:
            return JsonResponse({'message':'Sorry, could not submit this evaluation, please ensure you have answered all questions.','status':'error'})

    return render(request,'public/meeting_information/posters/poster_scoring.html', {'meeting':meeting, 'poster': poster, 'questions': questions })

@meeting_members_only
def meeting_poster_scoring_complete(request,meeting_slug,poster_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    poster = get_object_or_404(MeetingPoster, id=poster_id, meeting=meeting)

    return render(request,'public/meeting_information/posters/poster_scoring_complete.html',{'meeting':meeting, 'poster': poster})

@meeting_members_only
def meeting_exhibitors(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    exhibitors = MeetingExhibitor.objects.filter(meeting=meeting)

    return render(request,'public/meeting_information/exhibitors/exhibitors.html',{'exhibitors':exhibitors})

@meeting_members_only
def meeting_exhibitor(request,meeting_slug,exhibitor_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    exhibitor = get_object_or_404(MeetingExhibitor,id=exhibitor_id,meeting=meeting)

    user_note = False
    if request.user.is_authenticated():
        try:
            member = Member.objects.get(user=request.user)
            try:
                user_note = UserNote.objects.get(member=member,exhibitor=exhibitor)
            except:
                pass
        except:
            pass

    return render(request,'public/meeting_information/exhibitors/exhibitor-detail.html',{'exhibitor':exhibitor,'user_note':user_note})

@meeting_members_only
def meeting_exhibitor_note(request,meeting_slug,exhibitor_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    exhibitor = get_object_or_404(MeetingExhibitor,id=exhibitor_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,exhibitor=exhibitor)
    except:
        user_note = False

    return render(request,'public/meeting_information/exhibitors/exhibitor_note.html',{'meeting':meeting,'exhibitor':exhibitor,'user_note':user_note})

@meeting_members_only
def meeting_exhibitor_note_add(request,meeting_slug,exhibitor_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    exhibitor = get_object_or_404(MeetingExhibitor,id=exhibitor_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    try:
        user_note = UserNote.objects.get(member=member,exhibitor=exhibitor)
    except:
        user_note = False

    if request.POST:
        if user_note:
            user_note_form = UserNoteForm(request.POST,instance=user_note)
        else:
            user_note_form = UserNoteForm(request.POST)

        if user_note_form.is_valid():
            if user_note:
                user_note_form.save()
                return JsonResponse({'message':'Note has been updated.','status':'success'})
            else:
                user_note = user_note_form.save(commit=False)
                user_note.member = member
                user_note.exhibitor = exhibitor
                user_note.save()

                return JsonResponse({'message':'Note has been created.','status':'success'})

@meeting_members_only
def meeting_people(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    people = MeetingContact.objects.filter(meeting=meeting)

    return render(request,'public/meeting_information/people/people.html',{'people':people})

@meeting_members_only
def meeting_person(request,meeting_slug,person_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    contact = get_object_or_404(MeetingContact,id=person_id,meeting=meeting)

    chaired_sessions = ProgrammeSession.objects.filter(Q(chair=contact) | Q(chair_2=contact),meeting=meeting)
    speaker_presentations = SessionPresentation.objects.filter(Q(speaker=contact) | Q(speaker_2=contact) | Q(speaker_3=contact) | Q(speaker_4=contact)| Q(speaker_5=contact),meeting_session__meeting=meeting)
    posters = MeetingPoster.objects.filter(contact=contact,meeting=meeting)

    request.session['previous_url'] = reverse('meeting_person',args=[meeting.slug,contact.id])

    return render(request,'public/meeting_information/people/people-detail.html',{'person':contact,'chaired_sessions':chaired_sessions,'speaker_presentations':speaker_presentations,'posters':posters})


@meeting_members_only
def send_marketing_email(request, meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member, user=request.user)

    try:
        meeting_emailer = MeetingEmailer.objects.get(meeting=meeting, member=member)
    except MeetingEmailer.DoesNotExist:
        messages.error(request, 'Sorry your user cannot send emails')
        return redirect('meeting_home',meeting_slug=meeting.slug)

    meeting_message_form = MeetingMessageForm()
    registrants = MeetingBooking.objects.filter(meeting=meeting.meeting_booking)

    if request.POST:

        meeting_message_form = MeetingMessageForm(request.POST)

        if meeting_message_form.is_valid():

            memberids = request.POST.getlist('members')
            if memberids:
                submitted_members = []
                for memberid in memberids:
                    try:
                        submitted_member = MeetingBooking.objects.get(id=memberid, meeting=meeting.meeting_booking, complete=True)
                        submitted_members.append(submitted_member)
                    except:
                        pass
                
                if submitted_members:
                    meeting_message = meeting_message_form.save(commit=False)
                    meeting_message.meeting = meeting
                    meeting_message.sender = member
                    meeting_message.save()

                    recipients = []

                    for submitted_member in submitted_members:
                        message = MeetingMessageReceipient(message=meeting_message, receiver=submitted_member.member)
                        message.save()
                        recipients.append(submitted_member.email_address)

                    marketing_email(meeting_message, recipients)
                    messages.success(request, 'Your message has been sent')
                    return redirect('send_marketing_email', meeting_slug=meeting.slug)

                else:
                    messages.error(request, 'Please select members to send to')

            else:
                messages.error(request, 'Please select members to send to')

        else:
            print meeting_message_form.errors
            messages.error(request, 'Sorry, could not send your message - please try again below.')

    return render(request, 'public/meeting_information/email/send-email.html', {'meeting_message_form':meeting_message_form, 'registrants':registrants})

@meeting_members_only
def my_meeting(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    return render(request,'public/meeting_information/account/my_meeting.html',{})

@meeting_members_only
def my_messages(request, meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)

    meeting_messages = MeetingMessageReceipient.objects.filter(receiver=member)
    meeting_messages.annotate(orig_read=F('read'))

    for meeting_message in meeting_messages:
        meeting_message.read = True
        meeting_message.save()

    return render(request, 'public/meeting_information/messages/my-messages.html', {'meeting_messages':meeting_messages})


@meeting_members_only
def delete_message(request, meeting_slug, message_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)
    message = get_object_or_404(MeetingMessageReceipient, receiver=member, id=message_id)
    message.delete()

    messages.success(request, 'Message Deleted')
    return redirect(reverse('my_messages', args=[meeting.slug]))


@meeting_members_only
def my_programme(request,meeting_slug,day_id = False,time_slug = False):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)
    days = SessionDay.objects.filter(meeting=meeting)

    if day_id and time_slug:
        current_day = get_object_or_404(SessionDay,id=day_id,meeting=meeting)
        return render(request,'public/meeting_information/account/my_programme.html',{'days':days, 'current_day':current_day, 'current_time':time_slug})
    else:
        return render(request,'public/meeting_information/account/my_programme.html',{'days':days})

@meeting_members_only
def my_programme_load_day_time(request,meeting_slug,day_id,time_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)

    day = get_object_or_404(SessionDay,id=day_id,meeting=meeting)
    user_sessions = UserSession.objects.filter(member=member,session__meeting=meeting,session__time_of_day=time_slug,session__day=day)
    sessions = []
    for user_session in user_sessions:
        sessions.append(user_session.session)

    request.session['previous_url'] = reverse('my_programme_day_time',args=[meeting.slug,day.id,time_slug])

    return render(request,'public/meeting_information/account/day-time.html',{'day':day,'time_of_day':time_slug,'user_sessions':user_sessions,'sessions':sessions})

@meeting_members_only
def my_notes(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)

    session_notes       = UserNote.objects.filter(session__isnull=False,member=member)
    presentation_notes  = UserNote.objects.filter(presentation__isnull=False,member=member)
    poster_notes        = UserNote.objects.filter(poster__isnull=False,member=member)
    exhibitor_notes     = UserNote.objects.filter(exhibitor__isnull=False,member=member)

    return render(request,'public/meeting_information/account/my_notes.html',{'session_notes':session_notes,'presentation_notes':presentation_notes,'poster_notes':poster_notes,'exhibitor_notes':exhibitor_notes})

@meeting_members_only
def view_note(request,meeting_slug,note_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)

    note = get_object_or_404(UserNote,id=note_id,member=member)
    note_form = UserNoteForm(instance=note)

    return render(request,'public/meeting_information/account/view_note.html',{'note':note,'note_form':note_form})

@meeting_members_only
def change_note(request,meeting_slug,note_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)

    note = get_object_or_404(UserNote,id=note_id,member=member)

    if request.POST:
        note_form = UserNoteForm(request.POST,instance=note)

        if note_form.is_valid():
            note_form.save()

            return JsonResponse({'message':'Note has been updated','status':'updated'})

@meeting_members_only
def remove_note(request,meeting_slug,note_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)

    note = get_object_or_404(UserNote,id=note_id,member=member)
    note.delete()

    return JsonResponse({'message':'Note has been deleted','status':'deleted'})

@meeting_members_only
def meeting_page(request,meeting_slug,address):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    if address[-1] != '/':
        redirect_url = """%s/""" % (address)
        return redirect('meeting_page',meeting_slug=meeting.slug,address=redirect_url)

    parts = address.split('/')
    try:
        last_part = parts[-2]
    except:
        raise Http404

    try:
        page = MeetingPage.objects.get(url=last_part,meeting=meeting)
    except MeetingPage.DoesNotExist:
        raise Http404

    return render(request,'public/meeting_information/page.html',{'page':page})

@meeting_members_only
def bio_page(request, meeting_slug, bio_url):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    bio_page = get_object_or_404(MeetingBioPage, url=bio_url, meeting=meeting)

    return render(request, 'public/meeting_information/bio-page.html', {'page':bio_page})

@meeting_members_only
def meeting_search(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)


    if request.POST and request.POST.get('search_term'):

        search_term = request.POST['search_term']

        if search_term != '':
            sessions = ProgrammeSession.objects.filter(Q(name__icontains=search_term) | Q(programme_name__icontains=search_term) | Q(programme_description__icontains=search_term) | Q(chair__first_name__icontains=search_term) | Q(chair__last_name__icontains=search_term) | Q(chair_2__first_name__icontains=search_term) | Q(chair_2__last_name__icontains=search_term),meeting=meeting)
            presentations = SessionPresentation.objects.filter(Q(title__icontains=search_term) | Q(speaker__first_name__icontains=search_term) | Q(speaker__last_name__icontains=search_term) | Q(speaker_2__first_name__icontains=search_term) | Q(speaker_2__last_name__icontains=search_term) | Q(speaker_3__first_name__icontains=search_term) | Q(speaker_3__last_name__icontains=search_term) | Q(speaker_4__first_name__icontains=search_term) | Q(speaker_4__last_name__icontains=search_term) | Q(speaker_5__first_name__icontains=search_term) | Q(speaker_5__last_name__icontains=search_term), meeting_session__meeting=meeting)
            posters = MeetingPoster.objects.filter(Q(title__icontains=search_term) | Q(contact__first_name__icontains=search_term) | Q(contact__last_name__icontains=search_term), meeting=meeting)

            return render(request,'public/meeting_information/search.html',{'search_term':search_term,'sessions':sessions,'presentations':presentations,'posters':posters})

    return redirect('meeting_home',meeting_slug=meeting.slug)

@meeting_members_only
def voting_home(request,meeting_slug):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    member = get_object_or_404(Member,user=request.user)

    now = datetime.now()
    voting_sessions = MeetingVote.objects.filter(meeting=meeting,start_time__lte=now,end_time__gte=now)

    past_sessions = MeetingVote.objects.filter(meeting=meeting, end_time__lte=now)

    if member.member_type == 'non-member':
       voting_sessions = voting_sessions.filter(voting_type='all-users')
       past_sessions = past_sessions.filter(voting_type='all-users')
    else :
       if member.user_type != 'current':
            voting_sessions = voting_sessions.filter(voting_type='all-users')
            past_sessions = past_sessions.filter(voting_type='all-users')

    if voting_sessions.count() == 1 and not past_sessions:
        vote = voting_sessions[0]
        return redirect('vote',     meeting_slug=meeting.slug,vote_id=vote.id)

    return render(request,'public/meeting_information/vote/voting-home.html',{'voting_sessions':voting_sessions, 'past_sessions':past_sessions})

@meeting_members_only
def vote(request,meeting_slug,vote_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    now = datetime.now()
    vote = get_object_or_404(MeetingVote,id=vote_id,meeting=meeting,start_time__lte=now,end_time__gte=now)
    member = get_object_or_404(Member,user=request.user)

    if vote.voting_type == 'members-only':
        if member.user_type == 'non-member':
            return redirect('voting_home',meeting_slug=meeting.slug)
        else :
            if member.member_status != 'current':
               return redirect('voting_home',meeting_slug=meeting.slug)

    try:
        vote_submission = MeetingVoteSubmission.objects.get(vote=vote,member=member)

        if vote_submission.complete:
            return render(request,'public/meeting_information/vote/complete.html',{'vote':vote})

    except MeetingVoteSubmission.DoesNotExist:
        vote_submission = MeetingVoteSubmission(vote=vote,member=member)
        vote_submission.save()

    if vote.get_questions():
        first_question = vote.get_questions()[0]
        return redirect('vote_question',meeting_slug=meeting.slug,vote_id=vote.id,question_id=first_question.id)
    else:
        return render(request,'public/meeting_information/vote/no-questions.html',{'vote':vote})


@meeting_members_only
def vote_question(request,meeting_slug,vote_id,question_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    vote = get_object_or_404(MeetingVote,id=vote_id,meeting=meeting)
    question = get_object_or_404(MeetingVoteQuestion,id=question_id,vote=vote)
    member = get_object_or_404(Member,user=request.user)

    vote_submission = get_object_or_404(MeetingVoteSubmission,vote=vote,member=member)
    if vote_submission.complete:
        return redirect('vote',meeting_slug=meeting.slug,vote_id=vote.id)

    if request.POST and request.POST.get('answer'):

        try:
            question_submission = MeetingVoteQuestionSubmission.objects.get(vote_submission=vote_submission,question=question)
        except MeetingVoteQuestionSubmission.DoesNotExist:
            question_submission = MeetingVoteQuestionSubmission(vote_submission=vote_submission,question=question)

        question_submission.answer = request.POST['answer']
        question_submission.save()

        next_question = question.get_next_question()
        if next_question:
            return redirect('vote_question',meeting_slug=meeting.slug,vote_id=vote.id,question_id=next_question.id)
        else:
            vote_submission.complete = True
            vote_submission.save()

            return redirect('vote_complete',meeting_slug=meeting.slug,vote_id=vote.id)

    return render(request,'public/meeting_information/vote/question.html',{'vote':vote,'question':question})

@meeting_members_only
def vote_complete(request,meeting_slug,vote_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    vote = get_object_or_404(MeetingVote,id=vote_id,meeting=meeting)
    member = get_object_or_404(Member,user=request.user)

    vote_submission = get_object_or_404(MeetingVoteSubmission,vote=vote,member=member)
    if not vote_submission.complete:
        return redirect('vote',meeting_slug=meeting.slug,vote_id=vote.id)

    return render(request,'public/meeting_information/vote/complete.html',{'vote':vote})


@meeting_members_only
def vote_results(request, meeting_slug, vote_id):

    if request.user.is_staff:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug)
    else:
        meeting = get_object_or_404(MeetingInformation,slug=meeting_slug,enabled=True)

    now = datetime.now()
    vote = get_object_or_404(MeetingVote, id=vote_id,meeting=meeting)

    from django.utils import timezone
    if vote.end_time >= timezone.now():
        messages.warning(request, 'Sorry this voting has not taken place yet.')
        return redirect('voting_home', meeting_slug=meeting.slug)
    
    if not vote.show_results:
        messages.warning(request, 'Sorry you cannot see results for this vote.')
        return redirect('voting_home', meeting_slug=meeting.slug)

    if vote.voting_type == 'members-only':
        if member.user_type == 'non-member':
            return redirect('voting_home',meeting_slug=meeting.slug)
        else :
            if member.member_status != 'current':
               return redirect('voting_home',meeting_slug=meeting.slug)

    return render(request, 'public/meeting_information/vote/results.html', {'vote':vote })


def handler404(request):

    return render(request,'public/404.html',{},status=404)

def handler500(request):

    return render(request,'public/500.html',{},status=500)
