from django.db import models
from solo.models import SingletonModel
import base64


class Bio(models.Model):

    email_address = models.EmailField(blank=True)
    unique_key = models.CharField(max_length=200, blank=True)

    name = models.CharField(max_length=200, blank=True)
    description = models.TextField()
    image = models.ImageField(upload_to='meeting-information/bios', blank=True, null=True)

    details_entered = models.BooleanField(default=False)
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(blank=True, null=True)

    message = models.TextField(blank=True, null=True)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ['name']


class MeetingInformation(models.Model):

    name = models.CharField(max_length=200,verbose_name='Short Name')
    long_name = models.CharField(max_length=200,blank=True,null=True)
    slug = models.SlugField(max_length=200)
    enabled = models.BooleanField(default=False)
    content = models.TextField(blank=True)

    banner          = models.ImageField(upload_to='meeting-information',blank=True,null=True)
    mobile_banner   = models.ImageField(upload_to='meeting-information',blank=True,null=True)
    banner_title    = models.CharField(max_length=200,blank=True,null=True)
    banner_subtitle = models.CharField(max_length=200,blank=True,null=True)

    meeting_booking = models.ForeignKey('meeting_booking.Meeting', blank=True, null=True, help_text='Meeting booking to use for registrants.')

    abstract_request = models.ForeignKey('abstracts.AbstractRequest', blank=True,null=True,help_text='Abstract request to use for posters.')

    _pdf = models.TextField(blank=True)

    #evaluators
    evaluators = models.ManyToManyField('members.Member', blank=True)

    #splash page
    splash_enabled = models.BooleanField(default=False)
    splash_title = models.CharField(max_length=200, blank=True, null=True)
    splash_image = models.ImageField(upload_to='meeting-information', blank=True, null=True)
    splash_content = models.TextField(blank=True, null=True)

    def __unicode__(self):
        return self.name

    def set_pdf(self,data):
        self._pdf = base64.encodestring(data)

    def get_pdf(self):
        return base64.decodestring(self._pdf)

    pdf  = property(get_pdf,set_pdf)

    def get_all_pages(self):
        return MeetingPage.objects.filter(meeting=self)
    
    def get_bio_pages(self):
        return MeetingBioPage.objects.filter(meeting=self)

    def get_links(self):
        return MeetingLink.objects.filter(meeting=self)

    def get_programme_items(self):
        return ProgrammeSession.objects.filter(meeting=self)

    def get_contacts(self):
        return MeetingContact.objects.filter(meeting=self)

    def get_emailers(self):
        return MeetingEmailer.objects.filter(meeting=self)

    def get_posters(self):
        return MeetingPoster.objects.filter(meeting=self)

    def get_exhibitors(self):
        return MeetingExhibitor.objects.filter(meeting=self)

    def get_floor_plans(self):
        return MeetingFloorPlan.objects.filter(meeting=self)

    def get_banners(self):
        return MeetingBanner.objects.filter(meeting=self)

    def get_days(self):
        return SessionDay.objects.filter(meeting=self)
    
    def get_locations(self):
        return SessionLocation.objects.filter(meeting=self)
    
    def get_sessions(self):
        return ProgrammeSession.objects.filter(meeting=self)


class MeetingLink(models.Model):

    LINK_CHOICES = (
        (u'programme',u'Meeting Programme'),
        #(u'posters',u'Posters'),
        (u'exhibitors',u'Exhibitors'),
        (u'floor-plans', u'Floor Plans'),
        (u'my-programme',u'My Programme'),
        (u'bio-page',u'Bio Page'),
        (u'page',u'Page'),
        (u'message',u'Message (Sponsors)'),
        (u'voting', u'Voting'),
        (u'custom', u'Custom'),
        #(u'evaluation', u'Evaluation'),
    )

    ICON_CHOICES = (
        (u'programme', u'Programme'),
        (u'information', u'Information'),
        #(u'posters', u'Posters'),
        (u'exhibitors', u'Exhibitors'),
        (u'floor-plans', u'Floor Plans'),
        (u'my-programme', u'My Programme'),
        (u'voting', u'Voting'),
        (u'evaluation', u'Evaluation'),
        (u'camera', u'Camera'),
        (u'checkbox', u'Checkbox'),
        (u'document', u'Document'),
        (u'drinks', u'Drinks'),
        (u'info', u'Info (3)'),
        (u'info-2', u'Info (2)'),
        (u'pen-paper', u'Pen and Paper'),
        (u'plane', u'Plane'),
        (u'search', u'Search'),
        (u'spotlight', u'Spotlight'),
        (u'wifi', u'Wifi'),
        (u'yes-no', u'Yes / No'),
        (u'linkedin', u'Linked In'),
        (u'twitter', u'Twitter'),
        (u'youtube', u'Youtube'),
    )

    meeting = models.ForeignKey('MeetingInformation')

    name            = models.CharField(max_length=150)
    icon            = models.CharField(max_length=20,choices=ICON_CHOICES,blank=True,null=True)
    link_type       = models.CharField(max_length=100,choices=LINK_CHOICES,blank=True,null=True)
    page            = models.ForeignKey('MeetingPage',blank=True,null=True)
    bio_page        = models.ForeignKey('MeetingBioPage',blank=True,null=True)
    custom          = models.CharField(blank=True, null=True, max_length=300)
    order           = models.IntegerField(default=0)
    homepage        = models.BooleanField(default=False)
    navigation      = models.BooleanField(default=True)
    new_window      = models.BooleanField(default=False, verbose_name='Open in new window?')

    def link(self):
        from django.core.urlresolvers import reverse
        if self.link_type == 'programme':
            return '/%s/programme/' % (self.meeting.slug)
        elif self.link_type == 'posters':
            return '/%s/posters/' % (self.meeting.slug)
        elif self.link_type == 'exhibitors':
            return '/%s/exhibitors/' % (self.meeting.slug)
        elif self.link_type == 'my-programme':
            return '/%s/my-meeting/' % (self.meeting.slug)
        elif self.link_type == 'floor-plans':
            return '/%s/floor-plans/' % (self.meeting.slug)
        elif self.link_type == 'voting':
            return '/%s/voting/' % (self.meeting.slug)
        elif self.link_type == 'message':
            return '/%s/send-email/' % (self.meeting.slug)
        elif self.link_type == 'page':
            if self.page:
                return self.page.get_full_url()
        elif self.link_type == 'bio-page':
            if self.bio_page:
                return self.bio_page.get_full_url()
        elif self.link_type == 'custom':
            return self.custom
        return False

    def get_home_image(self):
        if self.icon:
            return 'images/microsite/home/%s.png' % (self.icon)
        else:
            return False

    def get_nav_image(self):
        if self.icon:
            return 'images/microsite/menu/%s.png' % (self.icon)
        else:
            return False

    def __unicode__(self):
        return self.name


    class Meta:
        ordering = ['order']

class MeetingPage(models.Model):

    meeting                 = models.ForeignKey('MeetingInformation')

    meta_title              = models.CharField(max_length=80, blank=True, help_text='The title to be displayed on search engines')
    meta_description        = models.CharField(max_length=160, blank=True, help_text='The description to be displayed on search engines')

    title                   = models.CharField(max_length=150)
    url                     = models.SlugField(max_length=150, blank=True, null=True)
    parent                  = models.ForeignKey('self', blank=True, null=True)
    introduction            = models.TextField(blank=True)
    introduction_image      = models.ImageField(upload_to='meeting-information/pages',blank=True,null=True)
    content                 = models.TextField(blank=True)

    class Meta:
        ordering = ['title']

    def __unicode__(self):
        return self.title

    def get_full_url(self):
        breadcrumbs = self.get_breadcrumbs()
        url = ''
        for breadcrumb in breadcrumbs:
            url = "%s%s/" % (url,breadcrumb.url)
        url = "/%s/%s" % (self.meeting.slug,url)
        return url

    def get_subpages(self):
        return MeetingPage.objects.filter(parent=self)

    def get_breadcrumbs(self):
        breadcrumbs = []
        breadcrumbs.append(self)

        if self.parent:
            parent_breadcrumbs = self.parent.get_parent_breadcrumbs(breadcrumbs)
            breadcrumbs.reverse()
            return breadcrumbs
        else:
            return breadcrumbs

    def get_parent_breadcrumbs(self,breadcrumbs):

        breadcrumbs.append(self)

        if self.parent:
            breadcrumbs = self.parent.get_parent_breadcrumbs(breadcrumbs)
            return breadcrumbs
        else:
            return breadcrumbs

    def get_top_level(self):
        if self.parent:
            top_level = self.parent.get_top_level()
            return top_level
        else:
            return self

    def get_resources(self):
        return MeetingPageResource.objects.filter(page=self)


class MeetingBioPage(models.Model):

    meeting                 = models.ForeignKey('MeetingInformation')

    meta_title              = models.CharField(max_length=80, blank=True, help_text='The title to be displayed on search engines')
    meta_description        = models.CharField(max_length=160, blank=True, help_text='The description to be displayed on search engines')

    title                   = models.CharField(max_length=150)
    url                     = models.SlugField(max_length=150,unique=True,blank=True,null=True)
    content                 = models.TextField(blank=True)

    bios                    = models.ManyToManyField('Bio', blank=True)

    class Meta:
        ordering = ['title']

    def __unicode__(self):
        return self.title

    def get_full_url(self):
        url = "/%s/profiles/%s/" % (self.meeting.slug,self.url)
        return url

class MeetingPageResource(models.Model):

    name = models.CharField(max_length=150)
    page = models.ForeignKey(MeetingPage)
    resource = models.ForeignKey('resources.Resource')
    order = models.IntegerField()

    def __unicode__(self):
        return self.name

class MeetingFloorPlan(models.Model):

    meeting = models.ForeignKey('MeetingInformation')

    name    = models.CharField(max_length=150)
    order   = models.IntegerField()
    image   = models.ImageField(upload_to='meeting_information/floor_plans')

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('order',)

class MeetingBanner(models.Model):

    meeting = models.ForeignKey('MeetingInformation')

    name    = models.CharField(max_length=150)
    order   = models.IntegerField()
    image   = models.ImageField(upload_to='meeting_information/banners')

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('order',)


class MeetingContact(models.Model):

    TITLE_CHOICES = (
        (u'Mr',u'Mr'),
        (u'Mrs',u'Mrs'),
        (u'Miss',u'Miss'),
        (u'Ms',u'Ms'),
        (u'Doctor',u'Doctor'),
        (u'Professor',u'Professor'),
    )

    meeting = models.ForeignKey('MeetingInformation')
    member = models.ForeignKey('members.Member',blank=True,null=True, on_delete=models.SET_NULL)
    title = models.CharField(max_length=20,blank=True,null=True)
    first_name = models.CharField(max_length=200)
    last_name = models.CharField(max_length=200)
    email_address = models.EmailField(blank=True,null=True)
    institution = models.CharField(max_length=200,blank=True,null=True)
    bio = models.TextField(blank=True,null=True)

    def __unicode__(self):
        return "%s %s" % (self.first_name,self.last_name)

    class Meta:
        ordering = ['last_name',]

class ProgrammeSession(models.Model):

    TIME_CHOICES = (
        (u'morning',u'Morning'),
        # (u'lunch',u'Lunch'),
        (u'afternoon',u'Afternoon'),
    )

    meeting = models.ForeignKey('MeetingInformation')
    name                    = models.CharField(max_length=200)
    programme_name          = models.CharField(max_length=200,blank=True)
    programme_description   = models.TextField(blank=True)
    programme_presenters    = models.TextField(blank=True)
    day = models.ForeignKey('SessionDay')
    start_time = models.TimeField()
    end_time = models.TimeField()
    time_of_day = models.CharField(max_length=20,choices=TIME_CHOICES) #auto generated on submission

    break_session = models.BooleanField(default=False,verbose_name='Break Session?')

    primary_subject = models.ForeignKey('SessionSubject',related_name='session_primary_subject',blank=True,null=True,on_delete=models.SET_NULL)
    secondary_subject = models.ForeignKey('SessionSubject',related_name='session_secondary_subject',blank=True,null=True,on_delete=models.SET_NULL)
    session_type = models.ForeignKey('SessionType',blank=True,null=True,on_delete=models.SET_NULL)
    location = models.ForeignKey('SessionLocation',blank=True,null=True,on_delete=models.SET_NULL)

    chair = models.ForeignKey('MeetingContact',blank=True,null=True,related_name="session_chair_1",on_delete=models.SET_NULL, verbose_name='Speaker')
    chair_2 = models.ForeignKey('MeetingContact',blank=True,null=True,related_name="session_chair_2",on_delete=models.SET_NULL, verbose_name='Speaker 2')

    content = models.TextField(blank=True,null=True)

    evaluation_enabled = models.BooleanField(default=False,verbose_name='Scoring Enabled')

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('meeting','day','start_time')

    def get_presentations(self):
        presentations = SessionPresentation.objects.filter(meeting_session=self)
        return presentations

    def get_time_of_day(self):
        import datetime
        if self.start_time < datetime.time(13,00):
            return 'morning'
        elif self.start_time < datetime.time(14,00):
            return 'lunch'
        else:
            return 'afternoon'
    
    def get_submissions(self):
        return SessionEvaluationSubmission.objects.filter(session=self)

    def save(self,*args,**kwargs):
        # self.time_of_day = self.get_time_of_day()
        super(ProgrammeSession, self).save(*args, **kwargs)

class SessionType(models.Model):

    name = models.CharField(max_length=200)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('name',)

class SessionSubject(models.Model):

    name = models.CharField(max_length=200)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('name',)

class SessionLocation(models.Model):

    meeting = models.ForeignKey('MeetingInformation')
    name = models.CharField(max_length=200)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('name',)

class SessionDay(models.Model):

    meeting = models.ForeignKey('MeetingInformation')
    date = models.DateField()

    def __unicode__(self):
        return self.date.strftime("%A %d %B")

    def d_format(self):
        return self.date.strftime("%d/%m/%Y")

    class Meta:
        ordering = ('date',)

    def get_sessions(self):
        sessions = ProgrammeSession.objects.filter(day=self,meeting=self.meeting)
        return sessions

class SessionPresentation(models.Model):

    title = models.CharField(max_length=200)
    meeting_session = models.ForeignKey('ProgrammeSession')
    order = models.IntegerField(default=0)
    speaker = models.ForeignKey('MeetingContact',blank=True,null=True,related_name='presentation_speaker',on_delete=models.SET_NULL)
    speaker_2 = models.ForeignKey('MeetingContact',blank=True,null=True,related_name='presentation_speaker_2',on_delete=models.SET_NULL)
    speaker_3 = models.ForeignKey('MeetingContact',blank=True,null=True,related_name='presentation_speaker_3',on_delete=models.SET_NULL)
    speaker_4 = models.ForeignKey('MeetingContact',blank=True,null=True,related_name='presentation_speaker_4',on_delete=models.SET_NULL)
    speaker_5 = models.ForeignKey('MeetingContact',blank=True,null=True,related_name='presentation_speaker_5',on_delete=models.SET_NULL)
    abstract_submission = models.ForeignKey('abstracts.AbstractSubmission',blank=True,null=True)

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ('meeting_session__day', 'meeting_session__start_time', 'meeting_session__id', 'order')

    def get_average_for_question(self, question):
        answers = SessionEvaluationSubmissionAnswer.objects.filter(presentation=self, question_name=question.name)
        if answers:
            score = 0
            for answer in answers:
                score += float(answer.answer)
            return score / answers.count()
        else:
            return 0

    def get_yesses_for_question(self, question):
        print question.name
        answers = SessionEvaluationSubmissionAnswer.objects.filter(presentation=self, question_name=question.name)
        if answers:
            score = 0
            for answer in answers:
                print answer.id
                if answer.answer == 'yes':
                    score += 1
            return score
        else:
            return 0

class MeetingPoster(models.Model):

    POSTER_TYPE_CHOICES = (
        (u'educational',u'Educational'),
        (u'scientific',u'Scientific'),
    )

    meeting = models.ForeignKey('MeetingInformation')

    title = models.CharField(max_length=200)
    poster_type = models.CharField(max_length=20,choices=POSTER_TYPE_CHOICES)
    abstract = models.TextField(blank=True)
    poster_file = models.FileField(upload_to='meeting-information/posters',blank=True,null=True)
    abstract_submission = models.ForeignKey('abstracts.AbstractSubmission',blank=True,null=True,on_delete=models.SET_NULL)
    contact = models.ForeignKey('MeetingContact',blank=True,null=True,on_delete=models.SET_NULL)

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ('title',)

    def get_submissions(self):
        return PosterEvaluationSubmission.objects.filter(poster=self)


class MeetingExhibitor(models.Model):

    meeting = models.ForeignKey('MeetingInformation')

    name = models.CharField(max_length=200)
    image = models.ImageField(upload_to='meeting-information/exhibitors',blank=True,null=True)
    description = models.TextField()
    booth = models.CharField(max_length=200)
    sponsor = models.BooleanField(default=False)
    floor_plan = models.ForeignKey('MeetingFloorPlan', blank=True, null=True)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('name',)

#user generated models
class UserSession(models.Model):

    member          = models.ForeignKey('members.Member')
    session         = models.ForeignKey('ProgrammeSession')
    created         = models.DateTimeField(auto_now_add=True)

    def __unicode__(self):
        return "%s - %s" % (self.member.user.get_full_name(),self.session)

class UserNote(models.Model):

    member          = models.ForeignKey('members.Member')
    session         = models.ForeignKey('ProgrammeSession',blank=True,null=True)
    presentation    = models.ForeignKey('SessionPresentation',blank=True,null=True)
    poster          = models.ForeignKey('MeetingPoster',blank=True,null=True)
    exhibitor       = models.ForeignKey('MeetingExhibitor',blank=True,null=True)
    created         = models.DateTimeField(auto_now_add=True)
    content         = models.TextField()

    def __unicode__(self):
        if self.session:
            return "%s - %s" % (self.member.user.get_full_name(),self.session)
        elif self.presentation:
            return "%s - %s" % (self.member.user.get_full_name(),self.presentation)
        elif self.poster:
            return "%s - %s" % (self.member.user.get_full_name(),self.poster)
        elif self.exhibitor:
            return "%s - %s" % (self.member.user.get_full_name(),self.exhibitor)
        else:
            return self.member.user.get_full_name()


class MeetingVote(models.Model):

    VOTING_TYPE_CHOICES = (
        (u'all-users',u'All Delegates'),
        (u'members-only',u'Members Only'),
    )

    meeting = models.ForeignKey('MeetingInformation')

    name = models.CharField(max_length=200)
    start_time = models.DateTimeField()
    end_time = models.DateTimeField()
    voting_type = models.CharField(max_length=200,default='all-users',choices=VOTING_TYPE_CHOICES)
    show_results = models.BooleanField(default=False)

    def __unicode__(self):
        return self.name

    def get_questions(self):
        return MeetingVoteQuestion.objects.filter(vote=self)


class MeetingVoteReminder(models.Model):

    vote = models.ForeignKey('MeetingVote')
    subject = models.CharField(max_length=200)
    content = models.TextField()
    to_addresses = models.TextField(blank=True)
    created = models.DateTimeField(auto_now_add=True)

    def __unicode__(self):
        return "%s - %s" % (self.vote, self.created)

class MeetingVoteQuestion(models.Model):

    vote = models.ForeignKey('MeetingVote')
    name = models.CharField(max_length=200)
    order = models.IntegerField()
    options = models.TextField()

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ('order',)

    def get_options(self):
        options = self.options.split(',')
        return options

    def get_next_question(self):
        questions = MeetingVoteQuestion.objects.filter(vote=self.vote,order__gt=self.order)
        if questions:
            return questions[0]
        else:
            return False

    def get_votes(self):
        all_submissions = MeetingVoteQuestionSubmission.objects.filter(question=self,vote_submission__complete=True)
        votes = []
        for option in self.get_options():
            answer_submissions = all_submissions.filter(answer=option)
            if answer_submissions:
                percentage = 100 * (float(answer_submissions.count()) / float(all_submissions.count()))
            else:
                percentage = 0
            votes.append({'name':option,'total_count':all_submissions.count(),'percentage':percentage,'count':answer_submissions.count()})

        return votes

class MeetingVoteSubmission(models.Model):

    member = models.ForeignKey('members.Member')
    vote = models.ForeignKey('MeetingVote')
    created = models.DateTimeField(auto_now_add=True)
    complete = models.BooleanField(default=False)

    def __unicode__(self):
        return "%s - %s" % (self.member.user.get_full_name(),self.vote)

class MeetingVoteQuestionSubmission(models.Model):

    vote_submission = models.ForeignKey('MeetingVoteSubmission')
    question = models.ForeignKey('MeetingVoteQuestion')
    answer = models.TextField()

    def __unicode__(self):
        return "%s - %s" % (self.vote_submission.member.user.get_full_name(),self.question)

class SessionEvaluationQuestion(models.Model):

    TYPE_CHOICES = (
        (u'scale',u'Scale (1-5)'),
        (u'text',u'Free Text')
    )

    meeting = models.ForeignKey('MeetingInformation')
    name = models.CharField(max_length=200)
    order = models.IntegerField()
    after = models.BooleanField(default=False,verbose_name='Display After Individual Session Questions')
    type = models.CharField(max_length=200,choices=TYPE_CHOICES)
    lower_scale = models.CharField(max_length=200,blank=True,null=True)
    upper_scale = models.CharField(max_length=200,blank=True,null=True)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ['order','after']

class PresentationEvaluationQuestion(models.Model):

    TYPE_CHOICES = (
        (u'scale',u'Scale (1-5)'),
        (u'yes-no',u'Yes / No'),
        (u'text',u'Free Text')
    )

    meeting = models.ForeignKey('MeetingInformation')
    name = models.CharField(max_length=200)
    order = models.IntegerField()
    type = models.CharField(max_length=200,choices=TYPE_CHOICES)
    lower_scale = models.CharField(max_length=200,blank=True,null=True)
    upper_scale = models.CharField(max_length=200,blank=True,null=True)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ['order']


class PosterEvaluationQuestion(models.Model):

    TYPE_CHOICES = (
        (u'scale',u'Scale (1-5)'),
        (u'yes-no',u'Yes / No'),
        (u'text',u'Free Text')
    )

    meeting = models.ForeignKey('MeetingInformation')
    name = models.CharField(max_length=200)
    order = models.IntegerField()
    type = models.CharField(max_length=200,choices=TYPE_CHOICES)
    lower_scale = models.CharField(max_length=200,blank=True,null=True)
    upper_scale = models.CharField(max_length=200,blank=True,null=True)

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ['order']

    def get_average_for_poster(self, poster):
        answers = PosterEvaluationSubmissionAnswer.objects.filter(question=self, evaluation_submission__poster=poster)
        if answers:
            score = 0
            for answer in answers:
                score += float(answer.answer)
            return score / answers.count()
        else:
            return 0

    def get_yesses_for_poster(self, poster):
        answers = PosterEvaluationSubmissionAnswer.objects.filter(question=self, evaluation_submission__poster=poster)
        if answers:
            score = 0
            for answer in answers:
                print answer.answer
                if answer.answer == 'yes':
                    score += 1
            return score
        else:
            return 0

class SessionEvaluationSubmission(models.Model):

    session = models.ForeignKey('ProgrammeSession')
    member = models.ForeignKey('members.Member')
    created = models.DateTimeField(auto_now=True)

    def __unicode__(self):
        return "%s - %s" % (self.session, self.member)

    class Meta:
        ordering = ['created',]

    def get_answers(self):
        return SessionEvaluationSubmissionAnswer.objects.filter(evaluation_submission=self)
    
    def get_presentation_answers(self, presentation):
        return SessionEvaluationSubmissionAnswer.objects.filter(presentation=presentation, evaluation_submission=self)

    def get_non_pres_answers(self):
        return SessionEvaluationSubmissionAnswer.objects.filter(presentation__isnull=True, evaluation_submission=self)

class SessionEvaluationSubmissionAnswer(models.Model):

    evaluation_submission = models.ForeignKey('SessionEvaluationSubmission')
    question = models.ForeignKey('SessionEvaluationQuestion', blank=True,null=True)
    presentation = models.ForeignKey('SessionPresentation', blank=True,null=True)
    question_name = models.CharField(max_length=200)
    answer = models.TextField()
    order = models.IntegerField()

    def __unicode__(self):
        return "%s %s" % (self.evaluation_submission,self.question_name)

    class Meta:
        ordering = ['order',]


class PosterEvaluationSubmission(models.Model):

    poster = models.ForeignKey('MeetingPoster')
    member = models.ForeignKey('members.Member')
    created = models.DateTimeField(auto_now=True)

    def __unicode__(self):
        return "%s - %s" % (self.poster, self.member)

    class Meta:
        ordering = ['created',]

    def get_answers(self):
        return PosterEvaluationSubmissionAnswer.objects.filter(evaluation_submission=self)


class PosterEvaluationSubmissionAnswer(models.Model):

    evaluation_submission = models.ForeignKey('PosterEvaluationSubmission')
    question = models.ForeignKey('PosterEvaluationQuestion', blank=True,null=True)
    question_name = models.CharField(max_length=200)
    answer = models.TextField()
    order = models.IntegerField()

    def __unicode__(self):
        return "%s %s" % (self.evaluation_submission,self.question_name)

    class Meta:
        ordering = ['order',]


class MeetingEmailer(models.Model):

    meeting = models.ForeignKey('MeetingInformation')
    member = models.ForeignKey('members.Member')

    def __unicode__(self):
        return "%s" % (self.member)

    def get_sent_messages(self):
        return MeetingMessage.objects.filter(sender=self.member)

    class Meta:
        ordering = ['meeting','member']

class MeetingMessage(models.Model):

    meeting = models.ForeignKey('MeetingInformation')
    sender = models.ForeignKey('members.Member', related_name='message_sender')
    created = models.DateTimeField(auto_now_add=True)

    subject = models.CharField(max_length=200)
    message = models.TextField()

    def __unicode__(self):
        return "%s" % (self.subject)

    class Meta:
        ordering = ['-created']

class MeetingMessageReceipient(models.Model):

    message = models.ForeignKey('MeetingMessage')
    receiver = models.ForeignKey('members.Member', related_name='message_receiver')
    read = models.BooleanField(default=False)

    def __unicode__(self):
        return "%s - %s" % (self.message.subject, self.receiver)



