#Django core bits
from django.shortcuts import render_to_response, get_object_or_404
from django.template import RequestContext, loader, Context
from django.contrib.auth.decorators import login_required, permission_required
from django.db.models import Q
from django.contrib.sites.models import Site
from django.conf import settings
from django.http import HttpResponseRedirect, Http404, HttpResponse
from django.core.urlresolvers import reverse
from django.contrib.auth.models import User, Group
from django.core.mail import EmailMessage
from django.contrib import messages

from datetime import datetime,date,time

from filetransfers.api import serve_file

#Forms
from adminforms import *

#Models
from models import *
from modules.meeting_booking.models import *
from modules.payments.models import *

#Others
import random,csv,string
from django.template.defaultfilters import slugify

from modules.core.decorators import *
from modules.core.functions import *
from signals import *
from functions import *

import pdfcrowd


#Admin Definitions
@permission_required('admin_users.can_access_members')
def admin_members(request):

    letters = map(chr, range(97, 123))
    member = Member()
    final_letters = []

    for letter in letters:
        status = member.get_letter_active(letter)
        if status:
            final_letters.append(letter)

    member_types = MemberType.objects.all().order_by('name')
    areas_of_interest = AreaOfInterest.objects.all().order_by('name')

    all_members = Member.objects.filter(approved=True,member_status='current',user_type='member').order_by('-user__date_joined')
    latest_members = Member.objects.filter(approved=True,member_status='current',user_type='member').order_by('-user__date_joined')[:10]
    suspended_members = Member.objects.filter(approved=True,member_status='suspended',user_type='member').order_by('surname')

    return render_to_response('admin/members/members.html', {'letters':letters,'final_letters':final_letters,'all_members':all_members,'latest_members': latest_members,'suspended_members':suspended_members,'member_types':member_types,'areas_of_interest':areas_of_interest}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def import_members(request):

    import_members_form = ImportMembersForm()

    if request.FILES:

        import_members_form = ImportMembersForm(request.POST,request.FILES)

        if import_members_form.is_valid():

            errors = False

            filename = 'members.csv'

            #try:
            handle_uploaded_file(import_members_form.cleaned_data['file'],filename)
            #except:
            #    errors = True
            #    messages.error(request,'Could not upload file')

            if not errors:

                #try:

                    csv_file = open("%s%s" % (settings.MEDIA_ROOT, filename), "rU")
                    reader = csv.reader(csv_file)

                    line_number = 1

                    for row in reader:

                        try:
                            if row[0] != 'First Name':

                                surname = row[1]
                                first_name = row[0]
                                email = row[2]

                                temp_email = False

                                if is_valid_email(email):
                                    member_email = email
                                else:
                                    member_email = "%s.%s@psgbimembers.org" % (first_name.lower(),surname.lower())
                                    temp_email = True

                                try:
                                    user = User.objects.get(email=member_email)

                                    messages.error(request,"%s Already Exists - Line %s" % (member_email,line_number))

                                except User.DoesNotExist:

                                    user = User(first_name=first_name,last_name=surname,email=member_email,is_active=False)
                                    user.username = random_string_unique(20,User,'username')
                                    generated_password = generate_random_password(10)
                                    user.set_password(generated_password)
                                    user.save()

                                    try:
                                        member = Member.objects.get(user__email = member_email)
                                        messages.error(request,"%s already exists - Line %s" % (member_email,line_number))
                                    except Member.DoesNotExist:

                                        member_id = row[0]

                                        try:
                                            member = Member.objects.get(membership_number=member_id)
                                            messages.error(request,"%s's ID already exists for another member - Line %s" % (member_email,line_number))
                                        except:

                                            member_type = row[1]

                                            try:
                                                member_type = MemberType.objects.get(id=1)
                                            except:
                                                member_type = None

                                            member = Member(
                                                user=user,
                                                approved=True,
                                                member_status='current',
                                                user_type='member',
                                                member_type=member_type,
                                                given_name= first_name,
                                                surname=surname,
                                                membership_number = get_next_membership_number(),
                                            )
                                            member.save()

                        except Exception, e:
                            messages.error(request,'Error on line %s %s' % (line_number,e))
                            raise e

                        line_number = line_number + 1

                    messages.success(request,'Members have been uploaded')

                    return HttpResponseRedirect(reverse('admin_members'))

                #except:
                #    messages.error(request,'Sorry, something was incorrect in your file upload. Please ensure you have entered all required columns fields and uploaded a CSV file.')

    return render_to_response('admin/members/import_members.html',{'import_members_form':import_members_form},context_instance=RequestContext(request))


@staff_only
def send_passwords(request):

    members = Member.objects.filter(user_type='member',member_status='current',requires_password_reset=True)
    #members = Member.objects.filter(user_type='member',member_status='current',requires_password_reset=True,user__email='steve@capabilitycloud.co.uk')

    for member in members:

        generated_password = generate_random_password(10)
        member.user.set_password(generated_password)
        member.user.is_active = True
        member.user.save()

        member.raw_password = generated_password
        member.temp_password = True
        member.save()

        #reset_password_notification = UserActivity(member=member,title='Please reset your password',text='You are currently using a temporary password, please <a href="%s">Change it to something you will remember</a>'% (reverse('change_password')),type='account-notification')
        #reset_password_notification.save()

        message = """

<p>Dear PSGBI Member,</p>

<p>I'm pleased to announce that the new website is now launched and available on our new address <a href="https://www.psgbi.org">www.psgbi.org</a>.</p>
<p>You are receiving this email as you currently appear on our database but it could be that your name and address are held in error and you may not be a member. However, you can still get access to the website as a member if you complete the registration form on arrival at the new site.</p>
<p>Once the trial period is over, you will be asked to renew your membership, at which time you can decide whether you wish to continue your membership, we hope you will continue to support the work of the Pancreatic Society of Great Britain and Ireland.</p>
<p>Below you will find a link to your account, click the link and once at the site, complete the registration form, most fields are mandatory; completing the form will enable us to ensure we have your latest information so we can better serve your needs.</p>
<p>We look forward to keeping you informed of developments within the Pancreatic Society and look forward to hearing from you about your needs from the new website
Please click the link below.</p>
<p>If you have any problems please email <a href="mailto:members@psgbi.org">members@psgbi.org</a></p>
<p></p>
<p>Kind regards,<br/>
Mr Andrew Smith<br/>
Secretary PSGBI</p>

<h2>Your Login Details</h2>

<p>Link: <a href="https://www.psgbi.org/login">https://www.psgbi.org/login</a><br/>
<p>Email Address: %s<br/>
Temporary password: %s</p>

""" % (member.user.email,generated_password)

        #msg = EmailMessage('PSGBI Website Member Registration', message, 'noreply@psgbi.org', ['daniel@calmdigital.com'])
        msg = EmailMessage('PSGBI Website Member Registration', message, 'noreply@psgbi.org', [member.user.email])
        msg.content_subtype = "html"

        try:
            msg.send()
        except:
            pass

    return HttpResponse('Ok')

@permission_required('admin_users.can_access_members')
def sort_subscriptions(request):

    members = Member.objects.all()

    for member in members:

        member_subscriptions = MemberSubscription.objects.filter(member=member).order_by('start_date')
        if member_subscriptions:
            start_date = member_subscriptions[0].start_date
            start_date_time = datetime(start_date.year,start_date.month,start_date.day,12,0,0)
            member.user.date_joined = start_date
            member.user.save()

            member.registered = start_date_time
            member.save()

            if member.user_type == 'member':
                reversed_subscriptions = member_subscriptions.reverse()
                expiry_date = reversed_subscriptions[0].expiry_date
                member.expiry_date = expiry_date
                member.save()

    return HttpResponse('Ok')


@permission_required('admin_users.can_access_members')
def sort_expired(request):

    december_2013 = date(2013,12,31)

    expired_members = Member.objects.filter(user_type='member',expiry_date__lte=december_2013)
    for member in expired_members:
        member.member_status = 'archived'
        member.save()

    return HttpResponse('Ok')

@permission_required('admin_users.can_access_members')
def admin_members_download(request):

    members = Member.objects.filter(approved=True,member_status='current',user_type='member').order_by('surname')

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=members.csv"

    writer = csv.writer(response)

    writer.writerow(['Membership Number','Title','Given Name','Surname','Email Address','Type','Registered', 'Expiry', 'Qualifications','Gender','Address 1','Address 2','Town','County','Country','Postcode','Telephone','Personal Email','Job Title','Hospital','Hospital Department','Hospital Address 1','Hospital Address 2','Hospital Town','Hospital County','Hospital Country','Hospital Postcode','Hospital Telephone','Hospital Email Address','Preferred Mailing Address'])

    for member in members:

        writer.writerow([unicode(s).encode("utf-8") for s in (member.membership_number,member.title,member.given_name,member.surname,member.user.email,member.member_type,member.user.date_joined, member.expiry_date, member.qualifications,member.get_gender_display(),member.address_1,member.address_2,member.town,member.county,member.country,member.postcode,member.telephone,member.personal_email,member.job_title,member.hospital,member.hospital_department,member.hospital_address_1,member.hospital_address_2,member.hospital_town,member.hospital_county,member.hospital_country,member.hospital_postcode,member.hospital_phone,member.hospital_email,member.get_preferred_mailing_address_display())])

    return response


@permission_required('admin_users.can_access_members')
def all_members(request):

    letters = map(chr, range(97, 123))
    members = Member.objects.filter(user_type='member',approved=True,member_status='current').order_by('user__last_name')
    member_types = MemberType.objects.all().order_by('name')

    return render_to_response('admin/members/all_members.html',{'letters':letters,'members':members,'member_types':member_types},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def alpha_members(request,letter):

    letters = map(chr, range(97, 123))
    member = Member()
    final_letters = []

    for prim_letter in letters:
        status = member.get_letter_active(prim_letter)
        if status:
            final_letters.append(prim_letter)

    member_types = MemberType.objects.all().order_by('name')
    areas_of_interest = AreaOfInterest.objects.all().order_by('name')

    members = Member.objects.filter(user__last_name__istartswith=letter,user_type='member',approved=True,member_status='current').order_by('user__last_name')

    return render_to_response('admin/members/member_list_alpha.html',{'letters':letters,'final_letters':final_letters,'member_types':member_types,'areas_of_interest':areas_of_interest,'members':members,'letter':letter,},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_member_list_type(request,id):

    letters = map(chr, range(97, 123))
    member = Member()
    final_letters = []

    for letter in letters:
        status = member.get_letter_active(letter)
        if status:
            final_letters.append(letter)

    member_types = MemberType.objects.all().order_by('name')
    areas_of_interest = AreaOfInterest.objects.all().order_by('name')

    member_type = get_object_or_404(MemberType,id=id)
    members = Member.objects.filter(member_type=member_type,user_type='member',approved=True,member_status='current').order_by('user__last_name')

    return render_to_response('admin/members/member_list_type.html',{'letters':letters,'final_letters':final_letters,'member_types':member_types,'areas_of_interest':areas_of_interest,'member_type':member_type,'members':members},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_member_list_type_download(request,id):

    member_type = get_object_or_404(MemberType,id=id)
    members = Member.objects.filter(member_type=member_type,user_type='member',approved=True,member_status='current').order_by('user__last_name')

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=members.csv"

    writer = csv.writer(response)

    writer.writerow(['Membership Number','Title','Given Name','Surname','Email Address','Type','Registered', 'Expiry Date', 'Qualifications','Gender','Address 1','Address 2','Town','County','Country','Postcode','Telephone','Personal Email','Job Title','Hospital','Hospital Department','Hospital Address 1','Hospital Address 2','Hospital Town','Hospital County','Hospital Country','Hospital Postcode','Hospital Telephone','Hospital Email Address','Preferred Mailing Address'])

    for member in members:

        writer.writerow([unicode(s).encode("utf-8") for s in (member.membership_number,member.title,member.given_name,member.surname,member.user.email,member.member_type,member.user.date_joined, member.expiry_date, member.qualifications,member.get_gender_display(),member.address_1,member.address_2,member.town,member.county,member.country,member.postcode,member.telephone,member.personal_email,member.job_title,member.hospital,member.hospital_department,member.hospital_address_1,member.hospital_address_2,member.hospital_town,member.hospital_county,member.hospital_country,member.hospital_postcode,member.hospital_phone,member.hospital_email,member.get_preferred_mailing_address_display())])

    return response

@permission_required('admin_users.can_access_members')
def admin_member_list_area(request,id):

    letters = map(chr, range(97, 123))
    member = Member()
    final_letters = []

    for letter in letters:
        status = member.get_letter_active(letter)
        if status:
            final_letters.append(letter)

    member_types = MemberType.objects.all().order_by('name')
    areas_of_interest = AreaOfInterest.objects.all().order_by('name')

    area_of_interest = get_object_or_404(AreaOfInterest,id=id)
    members = Member.objects.filter(special_interest_groups=sig,user_type='member',approved=True,member_status='current').order_by('user__last_name')

    return render_to_response('admin/members/member_list_sig.html',{'letters':letters,'final_letters':final_letters,'member_types':member_types,'areas_of_interest':areas_of_interest,'area_of_interest':area_of_interest,'members':members},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_member_list_area_download(request,id):

    area = get_object_or_404(AreaOfInterest,id=id)
    members = Member.objects.filter(area_of_interest=area,user_type='member',approved=True,member_status='current').order_by('user__last_name')

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=members.csv"

    writer = csv.writer(response)

    writer.writerow(['Membership Number','Title','Given Name','Surname','Email Address','Type','Registered', 'Expiry Date', 'Qualifications','Gender','Address 1','Address 2','Town','County','Country','Postcode','Telephone','Personal Email','Job Title','Hospital','Hospital Department','Hospital Address 1','Hospital Address 2','Hospital Town','Hospital County','Hospital Country','Hospital Postcode','Hospital Telephone','Hospital Email Address','Preferred Mailing Address'])

    for member in members:

        areas_of_interest = ''
        for area_of_interest in member.areas_of_interest.all():
            areas_of_interest = '%s%s, ' % (areas_of_interest,area_of_interest.name)

        writer.writerow([unicode(s).encode("utf-8") for s in (member.membership_number,member.title,member.given_name,member.surname,member.user.email,member.member_type,member.user.date_joined, member.expiry_date, member.qualifications,member.get_gender_display(),member.address_1,member.address_2,member.town,member.county,member.country,member.postcode,member.telephone,member.personal_email,member.job_title,member.hospital,member.hospital_department,member.hospital_address_1,member.hospital_address_2,member.hospital_town,member.hospital_county,member.hospital_country,member.hospital_postcode,member.hospital_phone,member.hospital_email,member.get_preferred_mailing_address_display())])

    return response


@permission_required('admin_users.can_access_members')
def archived(request):

    letters = map(chr, range(97, 123))
    member = Member()
    final_letters = []

    for letter in letters:
        status = member.get_letter_archived(letter)
        if status:
            final_letters.append(letter)

    members = Member.objects.filter(member_status='archived',user_type='member')[:20]

    return render_to_response('admin/members/archived-members.html',{'members':members,'letters':letters,'final_letters':final_letters},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def archived_alpha(request,letter):

    letters = map(chr, range(97, 123))
    member = Member()
    final_letters = []

    for prim_letter in letters:
        status = member.get_letter_archived(prim_letter)
        if status:
            final_letters.append(prim_letter)

    members = Member.objects.filter(user__last_name__istartswith=letter,member_status='archived',user_type='member').order_by('user__last_name')

    return render_to_response('admin/members/archived-members-alpha.html',{'letters':letters,'final_letters':final_letters,'members':members,'letter':letter},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def load_member(request):

    if request.POST:

        if request.POST.get('membership_number'):
            try:
                member = Member.objects.get(membership_number=request.POST['membership_number'], approved=True)

                return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except :
                messages.error(request,'Member with that number was not found.')
                return HttpResponseRedirect(reverse('admin_members'))
    else:
        messages.error(request,'Could not load member')

    return HttpResponseRedirect(reverse('admin_members'))


@permission_required('admin_users.can_access_members')
def search_name(request):

    letters = map(chr, range(97, 123))
    member = Member()
    final_letters = []

    for letter in letters:
        status = member.get_letter_active(letter)
        if status:
            final_letters.append(letter)

    member_types = MemberType.objects.all().order_by('name')
    areas_of_interest = AreaOfInterest.objects.all().order_by('name')

    if request.POST:
        if request.POST.get('member_name'):

            member_name = request.POST['member_name']
            members = Member.objects.filter(Q(given_name__icontains=request.POST['member_name']) | Q(surname__icontains=request.POST['member_name']), user_type='member', approved=True).order_by('surname')

            return render_to_response('admin/members/search-name.html',{'letters':letters,'final_letters':final_letters,'member_types':member_types,'areas_of_interest':areas_of_interest,'member_name':member_name,'members':members},context_instance=RequestContext(request))
        else:
            messages.error(request,'Could not search for members')
    else:
        messages.error(request,'Could not search for members')

    return HttpResponseRedirect(reverse('admin_members'))


@permission_required('admin_users.can_access_members')
def renewal_invoices(request):

    pending_invoices = MemberRenewalInvoice.objects.filter(paid=False).order_by('-created')

    return render_to_response('admin/members/invoices/invoices.html',{'pending_invoices':pending_invoices},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def download_renewal_invoices(request):

    pending_invoices = MemberRenewalInvoice.objects.filter(paid=False).order_by('-created')

@permission_required('admin_users.can_access_members')
def view_renewal_invoice(request,id):

    invoice = get_object_or_404(MemberRenewalInvoice,id=id)

    return render_to_response('admin/members/invoices/view-invoice.html',{'invoice':invoice},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def mark_paid_invoice(request,id):

    invoice = get_object_or_404(MemberRenewalInvoice,id=id,paid=False)
    member = invoice.member

    member_type = member.member_type
    current_expiry = member.expiry_date

    start_date = date.today()

    if member_type.subscription_length == '3-months':
        end_date = current_expiry+relativedelta(months=+3,days=-1)
    elif member_type.subscription_length == '6-months':
        end_date = start_date+relativedelta(months=+6,days=-1)
    else:
        end_date = start_date+relativedelta(months=+12,days=-1)

    member.expiry_date = end_date
    member.save()

    subscription_name = '%s Membership Subscription (%s)' % (member_type.name,start_date.strftime('%B %Y'))

    member_subscription = MemberSubscription(
        member=member,
        member_type=member_type,
        subscription=subscription_name,
        amount=member_type.renewal_fee,
        discount=0,
        total=member_type.renewal_fee,
        invoice_to=invoice.organisation_name,
        status='current',
        start_date=start_date,
        expiry_date=end_date,
        invoice_created=date.today())
    member_subscription.save()

    receipt_name = '%s Membership Subscription (%s)' % (settings.WEBSITE_NAME,start_date.strftime('%B %Y'))
    receipt = Receipt(member=member,name=receipt_name,start_date=start_date,amount_paid=member_type.renewal_fee,payment_type='Invoice')
    receipt.save()

    try:
        client = pdfcrowd.Client("calmdigital", "dc33e7c4525620565185d0a00d90b8f0")
        client.setPageMargins('50','50','0','50')
        output_file = open(os.path.join(settings.MEDIA_ROOT, 'members/receipts/membership_renewal_%s.pdf' % (receipt.id)), 'wb')
        html = render_to_string('members/receipts/view-receipt-pdf.html',{'receipt':receipt})
        client.convertHtml(html,output_file)
        output_file.close()
    except:
        pass

    receipt.file = 'members/receipts/membership_renewal_%s.pdf' % (receipt.id)
    receipt.save()

    #create certificate pdf
    certificate = Certificate(member=member,start_date=member_subscription.start_date,end_date=member_subscription.expiry_date)
    certificate.save()

    try:
        client = pdfcrowd.Client("calmdigital", "dc33e7c4525620565185d0a00d90b8f0")
        client.setPageMargins('50','50','0','50')
        output_file = open(os.path.join(settings.MEDIA_ROOT, 'members/certificates/membership_certificate_%s.pdf' % (certificate.id)), 'wb')
        html = render_to_string('members/certificates/view-cert-pdf.html',{'certificate':certificate},context_instance=RequestContext(request))
        client.convertHtml(html,output_file)
        output_file.close()
    except:
        pass

    certificate.file = 'members/certificates/membership_certificate_%s.pdf' % (certificate.id)
    certificate.save()

    renewal_confirmation.send(sender=None,member=member,subscription=member_subscription,method='Invoice',receipt=receipt,certificate=certificate,notify_admin=False)

    notification = UserActivity(member=member,type='account-notification',title='Membership Renewed',text='Your membership has been successfully renewed.<br/>Visit <a href="%s">My Receipts</a> to view your receipt.' % (reverse('my_receipts')))
    notification.save()

    invoice.paid = True
    invoice.save()

    messages.success(request,'Invoice has been marked as paid, Member has been notified')

    return HttpResponseRedirect(reverse('admin_members_renewal_invoices'))

@permission_required('admin_users.can_access_members')
def delete_renewal_invoice(request,id):

    invoice = get_object_or_404(MemberRenewalInvoice,id=id)
    try:
        invoice.delete()
        messages.success(request,'Invoice Deleted')

    except:
        messages.error(request,'Could not delete invoice')

    return HttpResponseRedirect(reverse('admin_members_renewal_invoices'))


@permission_required('admin_users.can_access_members')
def admin_add_member(request):

    member_user_form = AdminMemberUserForm()
    member_password_form = AdminAddMemberPasswordForm()
    member_form = AdminMemberForm()
    gift_aid_form = AdminMemberGiftAidForm()
    invoice_form  = AdminMemberInvoiceForm()

    if request.POST:
        member_user_form = AdminMemberUserForm(request.POST)
        member_password_form = AdminAddMemberPasswordForm(request.POST)
        member_form = AdminMemberForm(request.POST,request.FILES)
        gift_aid_form = AdminMemberGiftAidForm(request.POST)
        invoice_form  = AdminMemberInvoiceForm(request.POST)

        try:
            member = Member.objects.get(user__email=request.POST['email'])

            if member.member_type == 'delegate':
                member_type = 'A Delegate'
                member_url = reverse('admin_members_view_delegate',args=[member.id])
            else:
                member_url = reverse('view_member',args=[member.id])
                member_type = member.get_member_status_display()

            messages.error(request,'Email Address already exists, user is %s <a href="%s">View Profile</a>' %  (member_type,member_url))

        except Member.DoesNotExist:
            pass

        if member_user_form.is_valid() and member_password_form.is_valid() and member_form.is_valid() and gift_aid_form.is_valid() and invoice_form.is_valid():

            try:

                user = member_user_form.save(commit=False)
                user.set_password(request.POST['password1'])
                user.username = random_string_unique(20,User,'username')
                user.save()

                member = member_form.save(commit=False)
                member.user = user
                member.approved = True
                member.complete = True
                member.paid = True
                member.member_status = 'current'
                member.membership_number = get_next_membership_number()
                member.save()

                gift_aid_form = AdminMemberGiftAidForm(request.POST,instance=member)
                invoice_form = AdminMemberInvoiceForm(request.POST,instance=member)

                if member.given_name:
                    user.first_name = member.given_name
                if member.surname:
                    user.last_name = member.surname
                user.save()

                messages.success(request,'Member Created')
                return HttpResponseRedirect(reverse('admin_members'))

            except Exception,e:
                messages.error(request,'Could not add Member %s' % (e))

        else:

            messages.error(request,'Could not save the details, please ensure all the fields have been entered correctly.')

    return render_to_response('admin/members/add_member.html',{'member_user_form':member_user_form,'member_password_form':member_password_form,'member_form':member_form,'gift_aid_form':gift_aid_form,'invoice_form':invoice_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_view_member(request,id):

    member = get_object_or_404(Member, id=id)

    payments = Payment.objects.filter(member=member).exclude(status='pending').order_by('-created')
    member_arrears = MemberArrears.objects.filter(member=member)

    member_direct_debits = MemberDirectDebit.objects.filter(member=member)
    meeting_bookings = MeetingBooking.objects.filter(member=member,complete=True).order_by('-time')[:5]
    member_contacts = MemberContact.objects.filter(member=member).order_by('-contact_date')[:5]
    member_committees = MemberCommittee.objects.filter(member=member).order_by('-end_date')[:5]
    member_classifications = MemberClassification.objects.filter(member=member).order_by('-class_date')[:5]
    member_subscriptions = MemberSubscription.objects.filter(member=member).order_by('-expiry_date')[:5]

    return render_to_response('admin/members/view-member.html',{'member':member,'payments':payments,'member_direct_debits':member_direct_debits,'meeting_bookings':meeting_bookings,'member_events':member_events,'member_contacts':member_contacts,'member_committees':member_committees,'member_classifications':member_classifications,'member_subscriptions':member_subscriptions}, context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_edit_member(request,id):

    member = get_object_or_404(Member, id=id)

    member_user_form = AdminMemberUserForm(instance=member.user)
    member_password_form = AdminMemberPasswordForm()
    member_form = AdminMemberForm(instance=member)
    gift_aid_form = AdminMemberGiftAidForm(instance=member)
    invoice_form  = AdminMemberInvoiceForm(instance=member)

    initial_email = member.user.email

    if request.POST:
        member_user_form = AdminMemberUserForm(request.POST,instance=member.user)
        member_password_form = AdminMemberPasswordForm(request.POST)
        member_form = AdminMemberForm(request.POST,request.FILES,instance=member)
        gift_aid_form = AdminMemberGiftAidForm(request.POST,instance=member)
        invoice_form  = AdminMemberInvoiceForm(request.POST,instance=member)

        if member_user_form.is_valid() and member_form.is_valid() and gift_aid_form.is_valid() and invoice_form.is_valid():

            password_error = False
            new_password = False

            if request.POST.get('password1'):

                if not member_password_form.is_valid():
                    password_error = True
                new_password = request.POST['password1']

            if not password_error:

                member_user_form.save()

                if new_password:
                    member.user.set_password(request.POST['password1'])
                    member.user.save()

                member_form.save()

                if member.given_name:
                    member.user.first_name = member.given_name
                if member.surname:
                    member.user.last_name = member.surname

                member.user.save()

                if member.temp_email:
                    if  initial_email != member.user.email:
                        member.temp_email = False
                        member.save()

                gift_aid_form.save()
                invoice_form.save()

                messages.success(request,'Member Updated')
                return HttpResponseRedirect(reverse('view_member',args=[member.id]))

        else:
            messages.error(request,'Could not save the details, please ensure all the fields have been entered correctly.')

    return render_to_response('admin/members/edit_member.html',{'member':member,'member_user_form':member_user_form,'member_password_form':member_password_form,'member_form':member_form,'gift_aid_form':gift_aid_form,'invoice_form':invoice_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_delete_member(request, id):

    member = get_object_or_404(Member, id=id)
    try:
        member.user.delete()
        messages.success(request,'Member has been deleted')
    except:
        messages.error(request,'Member could not be deleted')

    return HttpResponseRedirect(reverse('admin_members'))

@permission_required('admin_users.can_access_members')
def admin_activate_member(request,id):
    member = get_object_or_404(Member,id=id)

    member.member_status = 'current'
    member.save()

    messages.success(request,'Member has been Re-Activated')

    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

@permission_required('admin_users.can_access_members')
def admin_archive_member(request,id):

    member = get_object_or_404(Member,id=id)

    member.member_status = 'archived'
    member.save()

    messages.success(request,'Member Archived')

    return HttpResponseRedirect(reverse('admin_members'))


@permission_required('admin_users.can_access_members')
def admin_suspend_member(request,id):

    member = get_object_or_404(Member,id=id)

    member.member_status = 'suspended'
    member.save()

    messages.success(request,'Member Marked as Suspended')

    return HttpResponseRedirect(reverse('admin_members_lapsed_members'))

@permission_required('admin_users.can_access_members')
def change_status(request,id,status):

    member = get_object_or_404(Member,id=id)
    today = date.today()

    if status == 'current':
        if member.member_status != 'current':
            member.member_status = 'current'
            member.save()
            messages.success(request,'Status Updated to <strong>Current</strong>')
        else:
            messages.warning(request,'Member is Already Current')

    elif status == 'lapsed':
        if member.member_status != 'lapsed':
            member.member_status = 'lapsed'
            member.lapsed_date = today
            member.save()
            messages.success(request,'Status Updated to <strong>Lapsed</strong>')
        else:
            messages.warning(request,'Member is Already Lapsed')

    elif status == 'suspended':
        if member.member_status != 'suspended':
            member.member_status = 'suspended'
            member.save()
            messages.success(request,'Status Updated to <strong>Suspended</strong>')
        else:
            messages.warning(request,'Member is Already Suspended')
    elif status == 'archived':
        if member.member_status != 'archived':
            member.member_status = 'archived'
            messages.success(request,'Status Updated to <strong>Archived</strong>')
            member.save()
        else:
            messages.warning(request,'Member is Already Archived')
    else:
        messages.error(request,'Could not change members status')

    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

@permission_required('admin_users.can_access_members')
def download_cv(request,id):

    member = get_object_or_404(Member,id=id)
    if member.cv_file:
        return serve_file(request, member.cv_file)
    else:
        raise Http404


@permission_required('admin_users.can_access_members')
def download_first_reference(request,id):

    member = get_object_or_404(Member,id=id)
    if member.reference_1_file:
        return serve_file(request, member.reference_1_file)
    else:
        raise Http404

@permission_required('admin_users.can_access_members')
def download_second_reference(request,id):

    member = get_object_or_404(Member,id=id)
    if member.reference_2_file:
        return serve_file(request, member.reference_2_file)
    else:
        raise Http404

@permission_required('admin_users.can_access_members')
def admin_add_member_arrears(request,id):

    member = get_object_or_404(Member,id=id)

    arrears_form = MemberArrearsForm()

    if request.POST:
        arrears_form = MemberArrearsForm(request.POST)

        if arrears_form.is_valid():

            #try:
            arrears = arrears_form.save(commit=False)
            arrears.member = member
            arrears.save()

            signals.member_arrears_created.send(sender=None,request=request,arrears=arrears)

            notification = UserActivity(member=member,type='account-notification',title='Arrears Payment Requested',text='Arrears are currently outstanding on your account - <a href="%s">pay these online now</a>.' % (reverse('account_pay_arrears',args=[arrears.id])))
            notification.save()

            messages.success(request,'Member Arrears Created')

            return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            #except:
            #    messages.error(request,'Could not create Member Arrears')

    return render_to_response('admin/members/arrears/add-member-arrears.html',{'member':member,'arrears_form':arrears_form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def admin_edit_member_arrears(request,id,arrears_id):

    member = get_object_or_404(Member,id=id)
    arrears = get_object_or_404(MemberArrears,id=arrears_id,member=member)

    arrears_form = MemberArrearsForm(instance=arrears)

    if request.POST:
        arrears_form = MemberArrearsForm(request.POST,instance=arrears)

        if arrears_form.is_valid():

            try:
                arrears = arrears_form.save()
                arrears.save()

                messages.success(request,'Member Arrears Updated')

                return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not update Member Arrears')

    return render_to_response('admin/members/arrears/edit-member-arrears.html',{'member':member,'arrears_form':arrears_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_delete_member_arrears(request,id,arrears_id):

    member = get_object_or_404(Member,id=id)
    arrears = get_object_or_404(MemberArrears,id=arrears_id,member=member)

    try:
        arrears.delete()
        messages.success(request,'Arrears have been deleted')
    except:
        messages.error(request,'Could not delete arrears')

    return HttpResponseRedirect(reverse('view_member',args=[member.id]))


@permission_required('admin_users.can_access_members')
def admin_update_paid_member_arrears(request,id,arrears_id):

    member = get_object_or_404(Member,id=id)
    arrears = get_object_or_404(MemberArrears,id=arrears_id,member=member)

    try:
        arrears.paid = True
        arrears.save()
        messages.success(request,'Arrears Updated')
    except:
        messages.error(request,'Could not Update Arrears')

    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

@permission_required('admin_users.can_access_members')
def member_transactions(request,id):

    member = get_object_or_404(Member, id=id)
    member_transactions = MemberTransaction.objects.filter(member=member).order_by('-transaction_date')

    return render_to_response('admin/members/transactions/all-transactions.html',{'member':member,'member_transactions':member_transactions}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def add_member_transaction(request,id):

    member = get_object_or_404(Member,id=id)
    transaction_form = MemberTransactionForm()

    if request.POST:
        transaction_form = MemberTransactionForm(request.POST)

        if transaction_form.is_valid():

            try:
                transaction = transaction_form.save(commit=False)
                transaction.member = member
                transaction.save()

                messages.success(request,'Transaction Added')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not add Transaction')

    return render_to_response('admin/members/transactions/add-transaction.html',{'member':member,'transaction_form':transaction_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def download_member_transactions(request,id):

    member = get_object_or_404(Member,id=id)
    transactions = MemberTransaction.objects.filter(member=member).order_by('-transaction_date')

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=%s-Transactions.csv" % (member.membership_number)

    writer = csv.writer(response)
    writer.writerow(['Name','Transaction Date', 'Invoice No', 'Item','Transaction Type','Batch','Amount','Vat','Balance','Paid','Payment Method','Printed','Credit Note Id'])

    for transaction in transactions:
        writer.writerow([unicode(s).encode("utf-8") for s in (transaction.name,transaction.transaction_date,transaction.invoice_no,transaction.item,transaction.transaction_type,transaction.batch,transaction.amount,transaction.vat,transaction.balance,transaction.paid,transaction.payment_method,transaction.printed,transaction.credit_note_id)])

    return response

@permission_required('admin_users.can_access_members')
def view_member_transaction(request,id,transaction_id):

    member = get_object_or_404(Member,id=id)
    member_transaction = get_object_or_404(MemberTransaction,id=transaction_id,member=member)

    return render_to_response('admin/members/transactions/view-transaction.html',{'member':member,'member_transaction':member_transaction},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def edit_member_transaction(request,id,transaction_id):

    member= get_object_or_404(Member,id=id)
    member_transaction = get_object_or_404(MemberTransaction,id=transaction_id,member=member)

    transaction_form = MemberTransactionForm(instance=member_transaction)

    if request.POST:

        transaction_form = MemberTransactionForm(request.POST,instance=member_transaction)

        if transaction_form.is_valid():

            try:
                transaction_form.save()

                messages.success(request,'Transaction has been saved')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not save transaction')

    return render_to_response('admin/members/transactions/edit-transaction.html',{'member':member,'member_transaction':member_transaction,'transaction_form':transaction_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_member_transaction(request,id,transaction_id):

    member = get_object_or_404(Member,id=id)
    member_transaction = get_object_or_404(MemberTransaction,id=transaction_id,member=member)

    try:
        member_transaction.delete()

        messages.success(request,'Transaction has been deleted')
    except:
        messages.error(request,'Could not delete transaction')

    if member.user_type == 'delegate':
        return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
    else:
        return HttpResponseRedirect(reverse('view_member',args=[member.id]))


@permission_required('admin_users.can_access_members')
def member_events(request,id):

    member = get_object_or_404(Member, id=id)
    member_events = MemberEvent.objects.filter(member=member).order_by('-created')

    return render_to_response('admin/members/events/all-events.html',{'member':member,'member_events':member_events}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def add_member_event(request,id):

    member = get_object_or_404(Member,id=id)
    event_form = MemberEventForm()

    if request.POST:
        event_form = MemberEventForm(request.POST)

        if event_form.is_valid():

            try:
                event = event_form.save(commit=False)
                event.member = member
                event.save()

                messages.success(request,'Event Added')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not add Event')

    return render_to_response('admin/members/events/add-event.html',{'member':member,'event_form':event_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def download_member_events(request,id):

    member = get_object_or_404(Member,id=id)
    events = MemberEvent.objects.filter(member=member)

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=%s-Events.csv" % (member.membership_number)

    writer = csv.writer(response)
    writer.writerow(['Event Name','Created', 'Member Name', 'Booking Type', 'Status'])

    for event in events:
        writer.writerow([unicode(s).encode("utf-8") for s in (event.event_name,event.created,event.member_name,event.booking_type,event.get_status_display())])

    return response

@permission_required('admin_users.can_access_members')
def view_member_event(request,id,event_id):

    member = get_object_or_404(Member,id=id)
    member_event = get_object_or_404(MemberEvent,id=event_id,member=member)

    return render_to_response('admin/members/events/view-event.html',{'member':member,'member_event':member_event},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def edit_member_event(request,id,event_id):

    member= get_object_or_404(Member,id=id)
    member_event = get_object_or_404(MemberEvent,id=event_id,member=member)

    event_form = MemberEventForm(instance=member_event)

    if request.POST:

        event_form = MemberEventForm(request.POST,instance=member_event)

        if event_form.is_valid():

            try:
                event_form.save()

                messages.success(request,'Event has been saved')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not save event')

    return render_to_response('admin/members/events/edit-event.html',{'member':member,'member_event':member_event,'event_form':event_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_member_event(request,id,event_id):

    member = get_object_or_404(Member,id=id)
    member_event = get_object_or_404(MemberEvent,id=event_id,member=member)

    try:
        member_event.delete()

        messages.success(request,'Event has been deleted')
    except:
        messages.error(request,'Could not delete event')

    if member.user_type == 'delegate':
        return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
    else:
        return HttpResponseRedirect(reverse('view_member',args=[member.id]))


@permission_required('admin_users.can_access_members')
def member_contacts(request,id):

    member = get_object_or_404(Member, id=id)
    member_contacts = MemberContact.objects.filter(member=member).order_by('-contact_date')

    return render_to_response('admin/members/contacts/all-contacts.html',{'member':member,'member_contacts':member_contacts}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def add_member_contact(request,id):

    member = get_object_or_404(Member,id=id)
    contact_form = MemberContactForm()

    if request.POST:
        contact_form = MemberContactForm(request.POST)

        if contact_form.is_valid():

            try:
                contact = contact_form.save(commit=False)
                contact.member = member
                contact.save()

                messages.success(request,'Contact Added')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not add Contact')

    return render_to_response('admin/members/contacts/add-contact.html',{'member':member,'contact_form':contact_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def download_member_contacts(request,id):

    member = get_object_or_404(Member,id=id)
    contacts = MemberContact.objects.filter(member=member)

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=%s-Contacts-.csv" % (member.membership_number)

    contact_date = models.DateField()
    contact_type = models.CharField(max_length=200)
    category = models.CharField(max_length=200)
    subject = models.CharField(max_length=200)
    operator = models.CharField(max_length=100)
    closed = models.BooleanField(default=False)

    writer = csv.writer(response)
    writer.writerow(['Contact Date','Contact Type','Category','Subject','Operator','Closed'])

    for contact in contacts:
        writer.writerow([unicode(s).encode("utf-8") for s in (contact.contact_date,contact.contact_type,contact.category,contact.subject,contact.operator,contact.closed)])

    return response


@permission_required('admin_users.can_access_members')
def view_member_contact(request,id,contact_id):

    member = get_object_or_404(Member,id=id)
    member_contact = get_object_or_404(MemberContact,id=contact_id,member=member)

    return render_to_response('admin/members/contacts/view-contact.html',{'member':member,'member_contact':member_contact},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def edit_member_contact(request,id,contact_id):

    member= get_object_or_404(Member,id=id)
    member_contact = get_object_or_404(MemberContact,id=contact_id,member=member)

    contact_form = MemberContactForm(instance=member_contact)

    if request.POST:

        contact_form = MemberContactForm(request.POST,instance=member_contact)

        if contact_form.is_valid():

            try:
                contact_form.save()

                messages.success(request,'Transaction has been saved')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not save contact')

    return render_to_response('admin/members/contacts/edit-contact.html',{'member':member,'member_contact':member_contact,'contact_form':contact_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_member_contact(request,id,contact_id):

    member = get_object_or_404(Member,id=id)
    member_contact = get_object_or_404(MemberContact,id=contact_id,member=member)

    try:
        member_contact.delete()

        messages.success(request,'Transaction has been deleted')
    except:
        messages.error(request,'Could not delete contact')

    if member.user_type == 'delegate':
        return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
    else:
        return HttpResponseRedirect(reverse('view_member',args=[member.id]))

@permission_required('admin_users.can_access_members')
def member_committees(request,id):

    member = get_object_or_404(Member, id=id)
    member_committees = MemberCommittee.objects.filter(member=member).order_by('-end_date')

    return render_to_response('admin/members/committees/all-committees.html',{'member':member,'member_committees':member_committees}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def add_member_committee(request,id):

    member = get_object_or_404(Member,id=id)
    committee_form = MemberCommitteeForm()

    if request.POST:
        committee_form = MemberCommitteeForm(request.POST)

        if committee_form.is_valid():

            try:
                committee = committee_form.save(commit=False)
                committee.member = member
                committee.save()

                messages.success(request,'Committee Added')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not add Committee')

    return render_to_response('admin/members/committees/add-committee.html',{'member':member,'committee_form':committee_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def download_member_committees(request,id):

    member = get_object_or_404(Member,id=id)
    committees = MemberCommittee.objects.filter(member=member)

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=%s-Committees-.csv" % (member.membership_number)

    writer = csv.writer(response)
    writer.writerow(['Committee','Position','Start Date','End Date'])

    for committee in committees:
        writer.writerow([unicode(s).encode("utf-8") for s in (committee.committee,committee.position,committee.start_date,committee.end_date)])

    return response
@permission_required('admin_users.can_access_members')
def view_member_committee(request,id,committee_id):

    member = get_object_or_404(Member,id=id)
    member_committee = get_object_or_404(MemberCommittee,id=committee_id,member=member)

    return render_to_response('admin/members/committees/view-committee.html',{'member':member,'member_committee':member_committee},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def edit_member_committee(request,id,committee_id):

    member= get_object_or_404(Member,id=id)
    member_committee = get_object_or_404(MemberCommittee,id=committee_id,member=member)

    committee_form = MemberCommitteeForm(instance=member_committee)

    if request.POST:

        committee_form = MemberCommitteeForm(request.POST,instance=member_committee)

        if committee_form.is_valid():

            try:
                committee_form.save()

                messages.success(request,'Transaction has been saved')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not save committee')

    return render_to_response('admin/members/committees/edit-committee.html',{'member':member,'member_committee':member_committee,'committee_form':committee_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_member_committee(request,id,committee_id):

    member = get_object_or_404(Member,id=id)
    member_committee = get_object_or_404(MemberCommittee,id=committee_id,member=member)

    try:
        member_committee.delete()

        messages.success(request,'Transaction has been deleted')
    except:
        messages.error(request,'Could not delete committee')

    if member.user_type == 'delegate':
        return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
    else:
        return HttpResponseRedirect(reverse('view_member',args=[member.id]))


@permission_required('admin_users.can_access_members')
def member_classifications(request,id):

    member = get_object_or_404(Member, id=id)
    member_classifications = MemberClassification.objects.filter(member=member).order_by('-class_date')

    return render_to_response('admin/members/classifications/all-classifications.html',{'member':member,'member_classifications':member_classifications}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def add_member_classification(request,id):

    member = get_object_or_404(Member,id=id)
    classification_form = MemberClassificationForm()

    if request.POST:
        classification_form = MemberClassificationForm(request.POST)

        if classification_form.is_valid():

            try:
                classification = classification_form.save(commit=False)
                classification.member = member
                classification.save()

                messages.success(request,'Classification Added')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not add Classification')

    return render_to_response('admin/members/classifications/add-classification.html',{'member':member,'classification_form':classification_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def download_member_classifications(request,id):

    member = get_object_or_404(Member,id=id)
    classifications = MemberClassification.objects.filter(member=member)

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=%s-Classifications-.csv" % (member.membership_number)

    writer = csv.writer(response)
    writer.writerow(['Parent Classification','Classification','Classification Date','Source'])

    for classification in classifications:
        writer.writerow([unicode(s).encode("utf-8") for s in (classification.parent_classification,classification.classification,classification.class_date,classification.source)])

    return response

@permission_required('admin_users.can_access_members')
def view_member_classification(request,id,classification_id):

    member = get_object_or_404(Member,id=id)
    member_classification = get_object_or_404(MemberClassification,id=classification_id,member=member)

    return render_to_response('admin/members/classifications/view-classification.html',{'member':member,'member_classification':member_classification},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def edit_member_classification(request,id,classification_id):

    member= get_object_or_404(Member,id=id)
    member_classification = get_object_or_404(MemberClassification,id=classification_id,member=member)

    classification_form = MemberClassificationForm(instance=member_classification)

    if request.POST:

        classification_form = MemberClassificationForm(request.POST,instance=member_classification)

        if classification_form.is_valid():

            try:
                classification_form.save()

                messages.success(request,'Transaction has been saved')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not save classification')

    return render_to_response('admin/members/classifications/edit-classification.html',{'member':member,'member_classification':member_classification,'classification_form':classification_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_member_classification(request,id,classification_id):

    member = get_object_or_404(Member,id=id)
    member_classification = get_object_or_404(MemberClassification,id=classification_id,member=member)

    try:
        member_classification.delete()

        messages.success(request,'Transaction has been deleted')
    except:
        messages.error(request,'Could not delete classification')

    if member.user_type == 'delegate':
        return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
    else:
        return HttpResponseRedirect(reverse('view_member',args=[member.id]))

@permission_required('admin_users.can_access_members')
def member_subscriptions(request,id):

    member = get_object_or_404(Member, id=id)
    member_subscriptions = MemberSubscription.objects.filter(member=member).order_by('-expiry_date')

    return render_to_response('admin/members/subscriptions/all-subscriptions.html',{'member':member,'member_subscriptions':member_subscriptions}, context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def add_member_subscription(request,id):

    member = get_object_or_404(Member,id=id)
    subscription_form = MemberAddSubscriptionForm()

    if request.POST:
        subscription_form = MemberAddSubscriptionForm(request.POST)

        if subscription_form.is_valid():

            #try:
            subscription = subscription_form.save(commit=False)
            subscription.member = member
            subscription.save()

            if subscription.renew_membership:
                member.expiry_date = subscription.expiry_date
                member.save()

                receipt_name = '%s Membership Subscription (%s)' % (settings.WEBSITE_NAME,subscription.start_date.strftime('%B %Y'))
                receipt = Receipt(member=member,name=receipt_name,start_date=subscription.start_date,amount_paid=subscription.total,payment_type=subscription.method)
                receipt.save()

                try:
                    client = pdfcrowd.Client("calmdigital", "dc33e7c4525620565185d0a00d90b8f0")
                    client.setPageMargins('50','50','0','50')
                    output_file = open(os.path.join(settings.MEDIA_ROOT, 'members/receipts/membership_renewal_%s.pdf' % (receipt.id)), 'wb')
                    html = render_to_string('members/receipts/view-receipt-pdf.html',{'receipt':receipt})
                    client.convertHtml(html,output_file)
                    output_file.close()
                except:
                    pass

                receipt.file = 'members/receipts/membership_renewal_%s.pdf' % (receipt.id)
                receipt.save()

                #create certificate pdf
                certificate = Certificate(member=member,start_date=subscription.start_date,end_date=subscription.expiry_date)
                certificate.save()

                try:
                    client = pdfcrowd.Client("calmdigital", "dc33e7c4525620565185d0a00d90b8f0")
                    client.setPageMargins('50','50','0','50')
                    output_file = open(os.path.join(settings.MEDIA_ROOT, 'members/certificates/membership_certificate_%s.pdf' % (certificate.id)), 'wb')
                    html = render_to_string('members/certificates/view-cert-pdf.html',{'certificate':certificate},context_instance=RequestContext(request))
                    client.convertHtml(html,output_file)
                    output_file.close()
                    certificate.file = 'members/certificates/membership_certificate_%s.pdf' % (certificate.id)
                    certificate.save()
                except:
                    pass

                renewal_confirmation.send(sender=None,request=request,member=member,subscription=subscription,method=subscription.method,receipt=receipt,certificate=certificate,notify_admin=False)

                notification = UserActivity(member=member,type='account-notification',title='Membership Renewed',text='Your membership has been successfully renewed.<br/>Visit <a href="%s">My Receipts</a> to view your receipt.' % (reverse('my_receipts')))
                notification.save()

                messages.success(request,'Subscription Added - Member has been renewed and notified')

            else:
                messages.success(request,'Subscription Added')


            if member.user_type == 'delegate':
                return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
            else:
                return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            #except:
            #    messages.error(request,'Could not add Subscription')

    return render_to_response('admin/members/subscriptions/add-subscription.html',{'member':member,'subscription_form':subscription_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def download_member_subscriptions(request,id):

    member = get_object_or_404(Member,id=id)
    subscriptions = MemberSubscription.objects.filter(member=member)

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = "attachment; filename=%s-Subscriptions-.csv" % (member.membership_number)

    writer = csv.writer(response)
    writer.writerow(['Subscription','Amount (GBP)','Discount (GBP)','Total (GBP)','Invoice To','Status','Start Date','Expiry Date','Invoice Created','Payment Method'])

    for subscription in subscriptions:
        writer.writerow([unicode(s).encode("utf-8") for s in (subscription.subscription,subscription.amount,subscription.discount,subscription.total,subscription.invoice_to,subscription.get_status_display(),subscription.start_date,subscription.expiry_date,subscription.invoice_created,subscription.method)])

    return response

@permission_required('admin_users.can_access_members')
def view_member_subscription(request,id,subscription_id):

    member = get_object_or_404(Member,id=id)
    member_subscription = get_object_or_404(MemberSubscription,id=subscription_id,member=member)

    return render_to_response('admin/members/subscriptions/view-subscription.html',{'member':member,'member_subscription':member_subscription},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def edit_member_subscription(request,id,subscription_id):

    member= get_object_or_404(Member,id=id)
    member_subscription = get_object_or_404(MemberSubscription,id=subscription_id,member=member)

    subscription_form = MemberEditSubscriptionForm(instance=member_subscription)

    if request.POST:

        subscription_form = MemberEditSubscriptionForm(request.POST,instance=member_subscription)

        if subscription_form.is_valid():

            try:
                subscription_form.save()

                messages.success(request,'Transaction has been saved')

                if member.user_type == 'delegate':
                    return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
                else:
                    return HttpResponseRedirect(reverse('view_member',args=[member.id]))

            except:
                messages.error(request,'Could not save subscription')

    return render_to_response('admin/members/subscriptions/edit-subscription.html',{'member':member,'member_subscription':member_subscription,'subscription_form':subscription_form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_member_subscription(request,id,subscription_id):

    member = get_object_or_404(Member,id=id)
    member_subscription = get_object_or_404(MemberSubscription,id=subscription_id,member=member)

    try:
        member_subscription.delete()

        messages.success(request,'Transaction has been deleted')
    except:
        messages.error(request,'Could not delete subscription')

    if member.user_type == 'delegate':
        return HttpResponseRedirect(reverse('admin_members_view_delegate',args=[member.id]))
    else:
        return HttpResponseRedirect(reverse('view_member',args=[member.id]))

@permission_required('admin_users.can_access_members')
def member_types(request):

    member_types = MemberType.objects.all().order_by('name')

    return render_to_response('admin/members/types/member-types.html',{'member_types':member_types},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def add_member_type(request):

    form = MemberTypeForm()

    if request.POST:
        form = MemberTypeForm(request.POST)

        if form.is_valid():
            try:
                form.save()
                messages.success(request,'Member Type Created')
            except:
                messages.error(request,'Could not add member type')

            return HttpResponseRedirect(reverse('admin_member_types'))

    return render_to_response('admin/members/types/add_member_type.html',{  'form':form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def edit_member_type(request,type_id):

    member_type = get_object_or_404(MemberType,id=type_id)

    form = MemberTypeForm(instance=member_type)

    if request.POST:
        form = MemberTypeForm(request.POST,request.FILES,instance=member_type)

        if form.is_valid():
            try:
                form.save()
                messages.success(request,'Member Type Updated')
            except:
                messages.error(request,'Could not update member type')

            return HttpResponseRedirect(reverse('admin_member_types'))

    return render_to_response('admin/members/types/edit_member_type.html',{'member_type':member_type,'form':form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_member_type(request,type_id):

    member_type = get_object_or_404(MemberType,id=type_id)

    try:
        member_type.delete()
        messages.success(request,'Member Type has been deleted')
    except:
        messages.error(request,'Could not delete member type')

    return HttpResponseRedirect(reverse('admin_member_types'))


@permission_required('admin_users.can_access_members')
def admin_member_groups(request):

    member_groups = MemberGroup.objects.all().order_by('order')

    return render_to_response('admin/members/groups/member_groups.html',{'member_groups':member_groups},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def admin_member_group_add(request):

    form = MemberGroupForm()

    if request.POST:
        form = MemberGroupForm(request.POST)

        if form.is_valid():
            #try:
            member_group = form.save()
            messages.success(request,'Member Group Added')
            #except:
            #messages.error(request,'Could not add member group')

            return HttpResponseRedirect(reverse('admin_member_groups'))

    return render_to_response('admin/members/groups/add_member_group.html',{'form':form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def admin_member_group_edit(request,group_id):

    member_group = get_object_or_404(MemberGroup,id=group_id)

    form = MemberGroupForm(instance=member_group)

    if request.POST:
        form = MemberGroupForm(request.POST,instance=member_group)

        if form.is_valid():
            #try:
            form.save()
            messages.success(request,'Member Group Updated')
            #except:
            #    messages.error(request,'Could not update member group')

            return HttpResponseRedirect(reverse('admin_member_groups'))

    return render_to_response('admin/members/groups/edit_member_group.html',{'member_group':member_group,'form':form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_member_group_delete(request,group_id):

    member_group = get_object_or_404(MemberGroup,id=group_id)

    try:
        member_group.delete()
        messages.success(request,'Member Group has been deleted')
    except:
        messages.error(request,'Could not delete member group')

    return HttpResponseRedirect(reverse('admin_member_groups'))


@permission_required('admin_users.can_access_members')
def areas_of_interest(request):

    areas_of_interest = AreaOfInterest.objects.all()

    return render_to_response('admin/members/areas_of_interest/areas_of_interest.html',{'areas_of_interest':areas_of_interest},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def add_area_of_interest(request):

    form = AreaOfInterestForm()

    if request.POST:
        form = AreaOfInterestForm(request.POST)

        if form.is_valid():
            try:
                form.save()
                messages.success(request,'Area of interest Added')
            except:
                messages.error(request,'Could not add Area of Interest')

            return HttpResponseRedirect(reverse('admin_members_areas_of_interest'))

    return render_to_response('admin/members/areas_of_interest/add_area_of_interest.html',{'form':form},context_instance=RequestContext(request))

@permission_required('admin_users.can_access_members')
def edit_area_of_interest(request,area_id):

    area = get_object_or_404(AreaOfInterest,id=area_id)

    form = AreaOfInterestForm(instance=area)

    if request.POST:
        form = AreaOfInterestForm(request.POST,instance=area)

        if form.is_valid():
            try:
                form.save()
                messages.success(request,'Area of Interest Updated')
            except:
                messages.error(request,'Could not update Area of Interest')

        return HttpResponseRedirect(reverse('admin_members_areas_of_interest'))

    return render_to_response('admin/members/areas_of_interest/edit_area_of_interest.html',{'area_of_interest':area_of_interest,'form':form},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def delete_area_of_interest(request,area_id):

    area = get_object_or_404(AreaOfInterest,id=area_id)

    try:
        area.delete()
        messages.success(request,'Area of Interest has been deleted')
    except:
        messages.error(request,'Could not delete Area of Interest')

    return HttpResponseRedirect(reverse('admin_members_areas_of_interest'))

@permission_required('admin_users.can_access_members')
def admin_payments(request):

    payments = Payment.objects.all().order_by('-created')

    return render_to_response('admin/members/payments/payments.html',{'payments':payments},context_instance=RequestContext(request))


@permission_required('admin_users.can_access_members')
def admin_view_payment(request,id,payment_id):

    payment = get_object_or_404(Payment,id=payment_id)

    return render_to_response('admin/members/payments/view-payment.html',{'payment':payment},context_instance=RequestContext(request))
