from django.db import models
from django.contrib.auth.models import User

from modules.pages.models import *
from modules.members.models import *

class ResourceCategory(models.Model):
    name        = models.CharField(max_length=50)
    slug        = models.SlugField(unique=True)
    description = models.TextField(blank=True)
    template    = models.CharField(max_length=150,blank=True,null=True)
    
    def __unicode__(self):
        return self.name
        
    def get_resources(self):
        resources = Resource.objects.filter(category=self)
        return resources
        
    def get_additional_fields(self):
        fields = ResourceAdditionalField.objects.filter(category=self)
        return fields
        
        
class ResourceAdditionalField(models.Model):

    TYPE_CHOICES = (
        (u'text',u'Text Field'),
        (u'file',u'File Field'),
        (u'image',u'Image Field'),
        (u'checkbox',u'Checkbox Field')
    )

    category    = models.ForeignKey(ResourceCategory)
    name        = models.CharField(max_length=100)
    slug        = models.CharField(max_length=100)
    type        = models.CharField(max_length=100,choices=TYPE_CHOICES)
    order       = models.IntegerField()
    
    def __unicode__(self):
        return self.name

class Resource(models.Model):
    name                    = models.CharField(max_length=200)
    category                = models.ForeignKey(ResourceCategory)
    slug                    = models.SlugField(unique=True)
    file                    = models.FileField(upload_to='resources', blank=True)
    description             = models.TextField(blank=True, verbose_name='Brief Description')
    embed_code              = models.TextField(blank=True)
    time                    = models.DateTimeField(auto_now_add=True)
    limit_to_member_types   = models.ManyToManyField('members.MemberType',blank=True)
    limit_to_member_groups  = models.ManyToManyField('members.MemberGroup',blank=True)
    author                  = models.CharField(max_length=100,blank=True,null=True)
	
    class Meta:
        ordering = ('name',)
    
    def __unicode__(self):
        return self.name
	
    def get_file_type(self):
        if self.file:
            fname, dot, extension = self.file.name.rpartition('.')

            if extension in ['jpg', 'gif', 'png', 'jpeg']:
            	return "Image"
	
            if extension in ['ppt','pptx']:
            	return "Presentation"
	
            if extension in ['pdf']:
            	return "PDF"
	
            if extension in ['doc','docx']:
            	return "Document"
	
            if extension in ['xls','xlsx']:
            	return "Spreadsheet"
	
            if extension in ['flv']:
            	return "Video"
        
        elif self.embed_code:
            return "Video"
        else:
            return False
		
    def get_additional_field_values(self):
        additional_field_values = ResourceAdditionalFieldValue.objects.filter(resource=self).order_by('additional_field__order')
        return additional_field_values
        
    def get_pages(self):
        
        pages = []
        page_resources = PageResource.objects.filter(resource=self)
        for page_resource in page_resources:
            page = page_resource.page
            pages.append(page)
        
        return pages
        
    def get_sections(self):
        
        sections = []
        section_resources = PageSectionResource.objects.filter(resource=self)
        for section_resource in section_resources:
            section = section_resource.section
            sections.append(section)
        return sections
        
    def get_permission(self,request):

        if request.user.is_authenticated():
            if request.user.is_superuser:
                return 'allowed'
            
            try:
                member = Member.objects.get(user=request.user)
                if member.is_current():
                    if self.limit_to_member_types.all():
                        if not member.member_type in self.limit_to_member_types.all():
                            return 'permission_denied'
                    if self.limit_to_member_groups.all():
                        in_groups = False
                        for group in self.limit_to_member_groups.all():
                            if member in group.members.all():
                                in_groups = True
            
                        if not in_groups:
                            return 'permission-denied'
                else:
                    if self.limit_to_member_types.all() or self.limit_to_member_groups.all():
                        return 'denied'
            except Member.DoesNotExist:
                if self.limit_to_member_groups.all() or self.limit_to_member_types.all():
                    return 'denied'
            
        else:
            if self.limit_to_member_groups.all() or self.limit_to_member_types.all():
                return 'denied'
        
        return 'allowed'

		
class ResourceAdditionalFieldValue(models.Model):
    
    resource = models.ForeignKey(Resource)
    additional_field = models.ForeignKey(ResourceAdditionalField)
    value = models.CharField(max_length=250,null=True,blank=True)
    file_value = models.FileField(max_length=250,upload_to='resources',blank=True,null=True)
    boolean_value = models.BooleanField(default=False)
    
    def __unicode__(self):
        return """%s-%s""" % (self.resource.name,self.additional_field.name)
    
    
    
    

